#!/bin/bash
#
# Usage: install_daemon.sh <imagefile>
#
# VERSION=31
# CHANGES="set nofpgareset when installing firmware"

# constants #
BEROCONF=/usr/fallback/beroconf
PIDFILE=/var/run/install_daemon.pid

[ -f /usr/fallback/helper ] && source /usr/fallback/helper

file=${1}
image_dir=/tmp/images

# functions #
function log {
	content="${2}"
	pipe=/tmp/install_pipe
	prefix="[install_${1:common}]"

	## show on GUI
	[ -p ${pipe} ] && echo "${prefix} $(date): ${content}" &>${pipe}

	if type -t permlog 2>&1 >/dev/null; then
		permlog "install-${1:common}" ${prefix} "${content}" ${3}
	else
		echo "${prefix} $(date): ${content}" | tee -a /usr/conf/permlog/install-${1:common}.log
	fi
}

function exit_daemon {
	## cleaning mtdblock7
	rm -rf /home/admin/conf
	mount -o remount,ro /usr/local
	mount -o remount,ro /home/admin
	## cleaning tmp directory
	for torm in "conf" "has.issue" "VERSION" "content.txt" "compare-content.txt" "setup"; do
		rm -rf /tmp/${torm}
	done
	sleep 5
	echo -n 2 > /sys/class/beronet/gateway/led_program
	rm -f ${PIDFILE}
	exit 1
}

# main #
if [ ! -d ${image_dir} ]; then
	mkdir -p ${image_dir}
fi

if [ -z "${file}" ] ; then
	log "init" "ERROR: No filename given, leaving." 1
	exit_daemon
fi

if [ ! -f "${image_dir}/${file}" ] ; then
	log "init" "ERROR: File ${image_dir}/${file} not found, leaving." 1
	exit_daemon
fi

if [ ! -L /mnt/images ]; then
	mount -o remount,rw /
	if [ -d /mnt/images ]; then
		rm -rf /mnt/images
	fi
	ln -s /tmp /mnt/images
	mount -o remount,ro /
fi

source /usr/local/VERSION

if [ ! -f /tmp/VERSION ]; then
	## should not happen since VERSION is untared from the fallback/install.sh script
	## should be removed in future fw
	if ! tar zxf ${image_dir}/${file} VERSION -C /tmp; then
		if ! tar xzf ${image_dir}/${file} tmp/VERSION -C / ; then
			if ! tar xzf ${image_dir}/${file} usr/local/VERSION -C /tmp ; then
				log "init" "ERROR: could not extract VERSION file" 1
				exit_daemon
			fi
		fi
	fi
fi

source /tmp/VERSION
[ -z ${APPFS} ] && APPFS=1

# check if TYPE-field is safe.
case "${TYPE}" in
	fpga|firmware|kernel|rootfs|appfsV3|userappfs)
		;;
	appfs|appfsV2)
		log "INVALID" "ERROR: This package can not be installed. Only 21.03 or higher" 1
		exit_daemon
		;;
	*)
		log "unknown" "ERROR: This package does not contain a valid type, leaving." 1
		exit_daemon
		;;
esac

[ -f /usr/conf/permlog/install-${TYPE:common}.log ] && rm -f /usr/conf/permlog/install-${TYPE:common}.log

log "${TYPE}" "Installing '${file}'." 1
case "${TYPE}" in
	fpga)
		# Put on a LED-Show for the customers!
		if [ -f /sys/class/beronet/gateway/led_program ]; then
			echo -n 3 > /sys/class/beronet/gateway/led_program
		fi

		log "${TYPE}" "Unpacking FPGA-Image."
		cd /tmp && tar zxf ${image_dir}/${file} fpga.img
		if [ ! -f /tmp/fpga.img ] ; then
			log "${TYPE}" "ERROR: Could not unpack the FPGA-Image, leaving."
			exit_daemon
		fi

		# retrieve Version-Information
		VERSION_INSTALLED=$(expr match "$(cat /sys/class/beronet/fpga/version)" "\([0-9]*\)")
		VERSION_TOINSTALL="$(expr match "$(cat /tmp/VERSION)" ".*FPGA=\(.*\)")"

		# check if fpga can be installed
		if [[ "$(/bin/cat /sys/class/beronet/gateway/li_count)" -eq "2" ]] && [[ "${VERSION_TOINSTALL}" -gt 9 ]]; then
			log "${TYPE}" "Installing a FPGA greater than VERSION 9 is not possible on this device"
			exit_daemon
		fi
		if [[ "$(/bin/cat /sys/class/beronet/gateway/li_count)" -eq "3" ]] && [[ "${VERSION_TOINSTALL}" -lt 12 ]]; then
			log "${TYPE}" "Installing a FPGA lower than VERSION 12 is not possible on this device"
			exit_daemon
		fi

		# tell the user about it
		log "${TYPE}" "Installing FPGA-Image. This process may take 3 minutes."
		log "${TYPE}" "Do NOT turn off your beroNet VoIP Gateway!"
		log "${TYPE}" "Updating FPGA-Image from Version ${VERSION_INSTALLED} to ${VERSION_TOINSTALL}."

		# log update-operations
		echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log

		# activate write-mode on fpga-eeprom
		echo 1 > /sys/class/beronet/fpga/eeprom
		if [ "$(cat /sys/class/beronet/fpga/eeprom)" -ne "1" ]; then
			log "${TYPE}" "ERROR: Could not enable write-access to FPGA-eeprom, leaving."
			exit_daemon
		fi

		# prepare the next reboot
		${BEROCONF} set root boot_fwupdate 0
		echo "1" > /tmp/reboot_after_update

		# Perform the Update
		sleep 2
		dd if=/tmp/fpga.img of=/dev/bfpga bs=512k count=1
		;;
	firmware)
		# Put on a LED-Show for the customers!
		if [ -f /sys/class/beronet/gateway/led_program ]; then
			echo -n 3 > /sys/class/beronet/gateway/led_program
		fi

		log "${TYPE}" "Unpacking Firmware."
		cd /tmp && tar zxf ${image_dir}/${file} firmware.axf
		if [ ! -f /tmp/firmware.axf ]; then
			log "${TYPE}" "ERROR: Could not unpack the Firmware-Image, leaving."
			exit_daemon
		fi

		# retrieve Version-Information
		VERSION_INSTALLED="unknown"
		if [ -f /usr/conf/VERSION.firmware ]; then
			VERSION_INSTALLED="$(expr match "$(cat /usr/conf/VERSION.firmware)" ".*FIRMWARE=\(.*\)")"
		fi
		VERSION_TOINSTALL="$(expr match "$(cat /tmp/VERSION)" ".*FIRMWARE=\(.*\)")"

		log "${TYPE}" "Installing Firmware-Image. This process may take 3 minutes."
		log "${TYPE}" "Do NOT turn off your beroNet VoIP Gateway!"
		log "${TYPE}" "Updating Firmware-Image from Version ${VERSION_INSTALLED} to ${VERSION_TOINSTALL}."

		# prepare the next reboot
		${BEROCONF} set root boot_fwupdate 0
		echo "1" > /tmp/reboot_after_update

		# log update-operations
		echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log
		mv /tmp/VERSION /usr/conf/VERSION.firmware

		# Perform the Update
		sleep 2
		dd if=/tmp/firmware.axf of=/dev/mtdblock3
	;;
	kernel)
		# Put on a LED-Show for the customers!
		if [ -f /sys/class/beronet/gateway/led_program ]; then
			echo -n 3 > /sys/class/beronet/gateway/led_program
		fi

		log "${TYPE}" "Unpacking Kernel-Image."
		cd /tmp && tar zxf ${image_dir}/${file} zImage
		if [ ! -f /tmp/zImage ]; then
			log "${TYPE}" "ERROR: Could not unpack the Kernel-Image, leaving."
			exit_daemon
		fi

		# Retrieve Version-Information
		VERSION_INSTALLED="unknown"
		if [ -f /usr/conf/VERSION.${TYPE} ]; then
			VERSION_INSTALLED="$($(expr match "$(cat /usr/conf/VERSION.kernel)" ".*KERNEL=\(.*\)"))"
		fi
		VERSION_TOINSTALL="$($(expr match "$(cat /tmp/VERSION)" ".*KERNEL=\(.*\)"))"

		log "${TYPE}" "Installing Kernel-Image. This process may take 3 minutes."
		log "${TYPE}" "Do NOT turn off your beroNet VoIP Gateway!"
		log "${TYPE}" "Updating Kernel-Image from Version ${VERSION_INSTALLED} to ${VERSION_TOINSTALL}."

		# log update-operations
		echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log
		mv /tmp/VERSION /usr/conf/VERSION.kernel

		# prepare the next reboot
		${BEROCONF} set root boot_fwupdate 0
		echo "1" > /tmp/reboot_after_update

		sleep 2
		dd if=/tmp/zImage of=/dev/mtdblock4
	;;
	rootfs)
		# Put on a LED-Show for the customers!
		if [ -f /sys/class/beronet/gateway/led_program ]; then
			echo -n 3 > /sys/class/beronet/gateway/led_program
		fi

		VERSION_INSTALLED="unknown"
		if [ -f /VERSION ]; then
			VERSION_INSTALLED=$(expr match "$(cat /VERSION)" ".*ROOTFS=\(.*\)")
		fi
		VERSION_TOINSTALL=$(expr match "$(cat /tmp/VERSION)" ".*ROOTFS=\(.*\)")

		echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log

		if [ -f /etc/ssh_host_key ]; then
			log "${TYPE}" "Saving SSH-Leys to /usr/conf."
			[ ! -d /usr/conf/ssh ] && mkdir /usr/conf/ssh
			cp -a /etc/ssh_host* /usr/conf/ssh/
		fi

		log "${TYPE}" "Stopping services."
		
		#no fpga reset when updating firmware
		echo nofpgareset > /tmp/nofpgareset

		/etc/init.d/S90app stop
		/etc/init.d/S60httpd stop

		# prepare the next reboot
		${BEROCONF} set root boot_fwupdate 0

		log "${TYPE}" "Loading berofix module."
		insmod /usr/local/modules/berofix.ko

		log "${TYPE}" "Extracting RootFS-Image."
		rm -rf /tmp/newroot
		mkdir /tmp/newroot || exit 1
		cd /tmp/newroot
		tar zxf ${image_dir}/${file} root.jffs2 || exit 1

		log "${TYPE}" "Saving RootFS to flash memory."
		dd if=root.jffs2 of=/dev/mtdblock5

		log "${TYPE}" "rebooting"
		echo 1 > /sys/class/beronet/fpga/reset
		;;
	userappfs)
		/usr/sbin/userapp_pkg_install.sh ${file}
		rm -f ${PIDFILE}
		;;
	appfsV3)
	
		# Put on a LED-Show for the customers!
		[ -f /sys/class/beronet/gateway/led_program ] && { echo -n 3 > /sys/class/beronet/gateway/led_program; }

		#no fpga reset when updating firmware
		echo nofpgareset > /tmp/nofpgareset

		## stop S90app and S91userapp in case dev run this script in normal mode
		[ -x /etc/init.d/S91userapp ] && /etc/init.d/S91userapp stop >/dev/null
		[ -x /etc/init.d/S90app ] && /etc/init.d/S90app stop >/dev/null

		##  stop cron
		#### on 20.06 / 21.01 / 21.02 like firmware
		if [ -x /etc/init.d/S45cron ]; then
			/etc/init.d/S45cron stop >/dev/null
		### on 21.03-like firmware
		elif [ -x /etc/init.d/S36cron ]; then
			/etc/init.d/S36cron stop >/dev/null
			## but restart it to have always the connection with berocloud (if cloud enabled)
			if [ "$(${BEROCONF} get root cloud_enable | grep -v failed)" = "1" ]; then
				/etc/init.d/S36cron start >/dev/null
				/etc/init.d/S65cloud start >/dev/null
			fi
		fi

		## if remotesupport running, stop it (to be removed in future
		[ -f /var/run/remotesupport ] && kill -9 $(cat /var/run/remotesupport)

		VERSION_INSTALLED="$(cat /usr/local/FILENAME)"

		log "${TYPE}" "Extracting install Scripts"
		if ! tar xzf ${image_dir}/${file} tmp/setup/pre_install.sh tmp/setup/install.sh -C / >/dev/null; then
			exit_daemon
		fi

		log "${TYPE}" "Executing pre_install-Script"
		if [ ! -x /tmp/setup/pre_install.sh ]; then
			log "${TYPE}" "pre_install-Script missing"
			exit_daemon
		fi
		/tmp/setup/pre_install.sh ${file}
		if [ -f /tmp/has.issue ]; then
			exit_daemon
		fi

		log "${TYPE}" "Executing install-Script"
		if [ ! -x /tmp/setup/install.sh ]; then
			log "${TYPE}" "install-Script missing"
			exit_daemon
		fi	
		/tmp/setup/install.sh ${file}
		if [ -f /tmp/has.issue ]; then
			exit_daemon
		fi

		## since post_install.sh script is used by S99finishprod. we keep the old behavior for this script
		log "${TYPE}" "Executing post_install-Script"
		if [ ! -x /usr/local/setup/post_install.sh ]; then
			log "${TYPE}" "post_install-Script missing. Leaving"
			## block pkg installation before exiting daemon
			mount -o remount,rw /usr/local
			rm -f /usr/local/conf/rootfs/VERSION.*
			mount -o remount,ro /usr/local
			exit_daemon
		fi	
		/usr/local/setup/post_install.sh

		VERSION_TOINSTALL="$(cat /usr/local/FILENAME)"

		${BEROCONF} set root boot_fwupdate 0
		${BEROCONF} set root app-image ${VERSION_TOINSTALL}
		${BEROCONF} delete root boot_recoverymode

		echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log

		#we always reboot after an appfs update now
		echo 1 > /tmp/reboot_after_update
		;;
esac

if [ -f /tmp/reboot_after_update ]; then
	log ${TYPE} "Rebooting beroNet VoIP Gateway to complete the Update."
fi

if [ -f /tmp/recover.device ]; then
	## install script executed from provisioning as recovery feature
	#### the tail proccess over install_pipe must be killed and the file removed
	#### since the device does not reboot automatically => we force it
	kill -9 $(ps | grep 'install_pipe' | awk '{ print $1; }')
	rm -f /tmp/install_pipe
fi

sync
sleep 1
rm -f ${PIDFILE}
exit 0
