#!/bin/bash
#
# VERSION=1
# CHANGES="script to update rootfs and switch from mtdblock5 to mtdblock7"

######## GLOBAL VARIABLES ########
ROOT_TAR=/tmp/images/$1

NEWPATH=/home/admin
WORKDIR=/tmp/update_rootfs
BEROCONF=/usr/fallback/beroconf
PIDFILE=/var/run/install_daemon.pid
FILES='root.jffs2 fw_printenv fw_setenv'
ETC_FILES='group hostname.default hosts lighttpd.pem moduli passwd shadow ssh_config sshd_config'
SSH_KEYS='authorized_keys known_hosts'

### create variables to linux command to avoid mixing between old and new rootfs
### set from old rootfs but can be used on both (via $NEWPATH for new rootfs)
WHICH=/usr/bin/which
BEROCONF=/usr/fallback/beroconf

CP=$($WHICH cp)
DD=$($WHICH dd)
DF=$($WHICH df)
LS=$($WHICH ls)
MV=$($WHICH mv)
RM=$($WHICH rm)
TR=$($WHICH tr)
WC=$($WHICH wc)
CAT=$($WHICH cat)
CUT=$($WHICH cut)
TAR=$($WHICH tar)
GREP=$($WHICH grep)
MKNOD=$($WHICH mknod)
MKDIR=$($WHICH mkdir)
MOUNT=$($WHICH mount)
UMOUNT=$($WHICH umount)
SQLITE2=$($WHICH sqlite)
BUSYBOX=$($WHICH busybox)

SQLITE3=/home/admin/usr/bin/sqlite3

######## FUNCTIONS ########
function log() {
        prefix="rootfs"
        logfile=/usr/conf/permlog/update-${prefix}.log
        pipe=/tmp/install_pipe

        if [ -p ${pipe} ]; then
                echo "[update_${prefix}] $(date): ${1}" &>${pipe}
        fi

        echo "[update_${prefix}] $(date): ${1}" >> ${logfile}
}

function restore_exit() {
	# restore recursively the berofix to the init state
	# from the last to the first modification and exit the script
	case "$1" in
		bootargslin)
			$WORKDIR/fw_setenv bootargs_lin console=ttyS0,115200 root=/dev/mtdblock5 rootfstype=jffs2 carduuid=$2 revision=$3 quiet &>/dev/null
			restore_exit "bootargsdbg" $2 $3
			;;
		bootargsdbg)
			$WORKDIR/fw_setenv bootargs_dbg console=ttyS0,115200 root=/dev/mtdblock5 rootfstype=jffs2 carduuid=$2 revision=$3 &>/dev/null
			restore_exit "devspace"
			;;
		devspace)
			$RM -f /dev/mtd1 &>/dev/null
			restore_exit "root"
			;;
		root)
			$MOUNT -o remount,ro / &>/dev/null
			restore_exit "aridb"
			;;
		aridb)
			$MV /usr/conf/oldari.db /usr/conf/ari.db &>/dev/null
			$RM -f /usr/conf/oldari.db &>/dev/null
			restore_exit "rootdb"
			;;
		rootdb)
			$MV /usr/conf/oldroot.db /usr/conf/root.db &>/dev/null
			$RM -f /usr/conf/oldroot.db &>/dev/null
			restore_exit "userapp"
			;;
		mkdrhomeadmin)
			## this case should not happen. secure scenario to try several times
			## to make the /home/admin directory
			if [ ! -d "/home/admin" ]; then
				i="0"
				while [[ ! -d "/home/admin" ]] && [[ $i -lt 10 ]]; do
					sync
					sleep 1
					$MKDIR -p /home/admin &>/dev/null
					i=$[$i+1]	
				done
			fi
			restore_exit "userapp"
			;;
		userapp)
			$MOUNT /home/admin &>/dev/null
			$MOUNT -o remount,rw /home/admin &>/dev/null
			$RM -rf /home/admin/* &>/dev/null
			if [ -f $WORKDIR/conf.tar.gz ]; then
				$TAR xzf $WORKDIR/conf.tar.gz -C /home/admin &>/dev/null
			fi
			$MOUNT -o remount,ro /home/admin &>/dev/null
			restore_exit "clean"
			;;
		clean)
			$MOUNT -o remount,ro / &>/dev/null
			$RM -rf $WORKDIR &>/dev/null
			restore_exit "initstate"
			;;
		initstate)
			$BEROCONF set root boot_fwupdate 1
			$BEROCONF delete root clean_oldrootfs
			;;
		*)
			;;
	esac
	log "Return to the Firmware-Update page in 5 seconds"
	sleep 5
	$RM -f $PIDFILE
	exit 1
}

function check_new_rootfs() {
	## check files
	if [[ ! -b "$NEWPATH/dev/mtdblock7" ]] || [[ ! -f "$NEWPATH$BUSYBOX" ]] || [[ ! -f "$NEWPATH/usr/bin/php-cgi" ]] || [[ ! -f "$SQLITE3" ]]; then
		log "Failed: missing rootfs files"
		restore_exit "userapp"
	fi
	## check /dev/mtdblock files
	for mtdblock_rights in $($LS -hl $NEWPATH/dev/mtdblock* | $CUT -d ' ' -f 0); do
		if [ ! "$mtdblock_rights" == "brw-r--r--" ]; then
			log "Failed: mtdblock rights incorrect"
			restore_exit "userapp"
		fi
	done
	## check busybox
	$NEWPATH$BUSYBOX --help &>/dev/null 
	if [ ! $? -eq 0 ]; then
		log "Failed: busybox does not work"
		restore_exit "userapp"
	fi
	## check php
	$NEWPATH/usr/bin/php-cgi --version &>/dev/null
	if [ ! $? -eq 0 ]; then
		log "Failed: php"
		restore_exit "userapp"
	fi
	## check sqlite
	SQLITE3_VERSION=$($SQLITE3 --version | $CUT -d ' ' -f 0)
	if [ ! "$SQLITE3_VERSION" == "3.11.0" ]; then
		log "Failed: sqlite3"
		restore_exit "userapp"
	fi
}

############ INIT ##########
if [[ -f "/usr/bin/sqlite3" ]] && [[ "$(/usr/bin/sqlite3 --version | $CUT -d ' ' -f 0)" == "3.11.0" ]]; then
	log "Rootfs already in the latest version"
	restore_exit
fi

log "Updating Rootfs. This process may take several minutes."
log "Do NOT turn off your beroNet VoIP Gateway !"
log "Init rootfs updating"
$BEROCONF set root boot_fwupdate 0
$BEROCONF set root clean_oldrootfs 1

## check the uploaded archive
$MKDIR $WORKDIR &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: make working directory space"
	restore_exit "clean"
fi	
$TAR xzf $ROOT_TAR -C $WORKDIR &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: untar archive"
	restore_exit "clean"
fi
for file in $FILES; do
	if [ ! -f "$WORKDIR/$file" ]; then
		log "Failed: file missing - $file"
		restore_exit "clean"
	fi
done
## get revision and serial
REVISION=/sys/class/beronet/gateway/revision
SERIAL=/sys/class/beronet/gateway/serial
if [ ! -f $REVISION ]; then
	log "Failed: revision file missing"
	restore_exit "clean"
fi
REVISION=$($CAT $REVISION)
if [ ! -f $SERIAL ]; then
	log "Failed: serial file missing"
	restore_exit "clean"
fi
SERIAL=$($CAT $SERIAL)

############ ERASE - clean the /dev/mtdblock7 partition ##########
log "Erasing userapp partition"
### make a backup if /home/admin space lower than 6 MB
if [ "$($DF | $GREP mtdblock7 | $TR -s ' ' | $CUT -d ' ' -f 3)" -lt 6000 ]; then
	if [ "$($LS -1 /home/admin | $WC -l | $TR -d ' ')" -gt 0 ]; then 
		$TAR cz -f $WORKDIR/conf.tar.gz -C /home/admin/ $($LS -1 /home/admin) &>/dev/null
		if [ ! $? -eq 0 ]; then
			log "Failed: make backup"
			restore_exit "clean"
		fi	
	fi
fi
$MOUNT -o remount,rw / &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: remount root space"
	restore_exit "clean"
fi
$MOUNT -o remount,rw /home/admin &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: remount userapp space"
	restore_exit "clean"
fi
$RM -rf /home/admin/* &>/dev/null
if [ ! $? -eq 0 ]; then
	log "failed: clean userapp space"
	restore_exit "userapp"
fi
$UMOUNT /home/admin &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: umount userapp space"
	restore_exit "userapp"
fi
$RM -rf /home/admin &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: remove userapp space"
	restore_exit "userapp"
fi
### final check 
if [ -d /home/admin ]; then
	log "Check failed: restoring the berofix"
	restore_exit "userapp"
fi

############ SET - install the new rootfs to /dev/mtdblock7 partition ##########
log "Installing new Rootfs"
$DD if=$WORKDIR/root.jffs2 of=/dev/mtdblock7 &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: paste new nootfs"
	restore_exit "userapp"
fi
$MKDIR /home/admin &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: make userapp space"
	restore_exit "mkdrhomeadmin"
fi
$MOUNT /home/admin &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: mount mtdblock7"
	restore_exit "userapp"
fi
### final check
check_new_rootfs

############ SAVE - keep relevant information #######
log "Saving admin settings"
$MOUNT -o remount,rw /home/admin &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: remount /home/admin"
	restore_exit "userapp"
fi
for file in $ETC_FILES; do
	if [ -f /etc/$file ]; then
		$CP /etc/$file /home/admin/etc/$file &>/dev/null
		if [ ! $? -eq 0 ]; then
			log "Failed: copy / paste $file (etc)"
			restore_exit "userapp"
		fi
	fi
done
for file in $SSH_KEYS; do
	if [ -f /root/.ssh/$file ]; then
		$RM -rf /home/admin/root/.ssh/$file &>/dev/null
		$CP /root/.ssh/$file /home/admin/root/.ssh/$file &>/dev/null
		if [ ! $? -eq 0 ]; then
			log "Failed: copy / paste $file (ssh)"
			restore_exit "userapp"
		fi
	fi
done

############ CONVERT - convert the DBs from sqlite2 to sqlite3 ##########
log "Converting the DBs"
## root
$MV /usr/conf/root.db /usr/conf/oldroot.db &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: backup root database"
	restore_exit "userapp"
fi
$SQLITE2 /usr/conf/oldroot.db .dump | $SQLITE3 /usr/conf/root.db &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: convert root database"
	restore_exit "rootdb"
fi
### final check
$SQLITE3 /usr/conf/root.db .dump &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Check failed root db"
	restore_exit "rootdb"
fi
## ari
$MV /usr/conf/ari.db /usr/conf/oldari.db &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: backup ari database"
	restore_exit "rootdb"
fi
$SQLITE2 /usr/conf/oldari.db .dump | $SQLITE3 /usr/conf/ari.db &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: convert ari database"
	restore_exit "aridb"
fi
### final check
$SQLITE3 /usr/conf/ari.db .dump &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Check failed ari db"
	restore_exit "aridb"
fi

############ U-BOOT - update u-boot to boot to /dev/mtdblock7 partition #########
if [ -c /dev/mtd1 ]; then
	$RM -f /dev/mtd1 &>/dev/null
	if [ ! $? -eq 0 ]; then
		log "Failed: removing mtd1"
		restore_exit "root"
	fi
fi
$MKNOD /dev/mtd1 c 90 2 &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: creating special file mtd1"
	restore_exit "root"
fi
## check
$WORKDIR/fw_printenv | $GREP mtdblock5 &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: bad CRC"
	restore_exit "devspace"
fi
## update u-boot env
$WORKDIR/fw_setenv bootargs_dbg console=ttyS0,115200 root=/dev/mtdblock7 rootfstype=jffs2 carduuid=$SERIAL revision=$REVISION &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: update bootargs_dbg"
	restore_exit "devspace"
fi
$WORKDIR/fw_printenv | $GREP mtdblock7 | $GREP bootargs_dbg &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: update bootargs_dbg"
	restore_exit "devspace"
fi
$WORKDIR/fw_setenv bootargs_lin console=ttyS0,115200 root=/dev/mtdblock7 rootfstype=jffs2 carduuid=$SERIAL revision=$REVISION quiet &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Failed: update bootargs_lin"
	restore_exit "bootargsdbg" $SERIAL $REVISION
fi
$WORKDIR/fw_printenv | $GREP mtdblock7 | $GREP bootargs_lin &>/dev/null
if [ ! $? -eq 0 ]; then
	log "Check failed"
	restore_exit "bootargslin" $SERIAL $REVISION
fi

############ CLEAN and REBOOT #######
log "Rebooting beroNet VoIP Gateway to complete the Update"
echo 1 > /tmp/reboot_after_update

sync
sleep 1
$RM -f /usr/conf/oldroot.db /usr/conf/oldari.db &>/dev/null
$RM -f ${PIDFILE}
exit 0
