#!/bin/bash
#
#	Install script during AppFs installation
#
# VERSION=1
# CHANGES=init

## PLEASE NOTE
##	mtdblock5 must be only readable
##	mtdblock6 and mtdblock7 must be always readable and writable

FW=${1}
IMAGE_DIR=/tmp/images

[ -f /usr/fallback/helper ] && source /usr/fallback/helper

source /tmp/VERSION
PREFIX=${TYPE:common}
LOGFILE=/usr/conf/permlog/install-install-${PREFIX}.log

### FUNCTIONS ###
function log() {
	pipe=/tmp/install_pipe
	output="[install_${PREFIX}] $(date): ${1}"

	## show on GUI
	[ -p ${pipe} ] && echo "${output}" &>${pipe}

 	## write on logs / show on cmdline
	if type -t permlog 2>&1 >/dev/null; then
		permlog "install-install-${PREFIX}" "[install_${PREFIX}]" "${1}"
	else
		echo "${output}" | tee -a ${LOGFILE}
	fi
}

function installFW() {
	#### PLEASE: Do not update the stderr and stdout pipe
	#### Function used to return only 0 and 1

	## Cleaning partitions
	log "Cleaning partitions" 2>&1 >/dev/null
	cd /usr/local && ls | grep -v lost | xargs rm -rf 2>/dev/null 1>/dev/null
	rm -rf /home/admin/conf 2>/dev/null 1>/dev/null
	rm -rf /tmp/conf 2>/dev/null 1>/dev/null

	## Extracting firmware
	log "Extracting new Firmware" 2>/dev/null 1>/dev/null
	if ! tar xzf ${IMAGE_DIR}/${FW} -C / 2>/dev/null 1>/dev/null ; then
		log "ERROR: Extracting new firmware failed" 2>&1 >/dev/null
		echo 1
	else
		## check /usr/local not empty and conf/rootfs exists
		if [ "$(ls /usr/local | wc | awk '{ print $1; }')" = "0" ]; then
			log "ERROR: Extracting files to appfs space failed" 2>&1 >/dev/null
			echo 1
		elif [ ! -d /tmp/conf/rootfs ]; then
			log "ERROR: Extracting files to tmp space failed" 2>&1 >/dev/null
			echo 1
		elif [ ! -d /home/admin/conf/rootfs ]; then
			log "ERROR: Extracting files to userapps space failed" 2>&1 >/dev/null
			echo 1
		else
			echo 0
		fi
	fi
}

function refreshSize() {
	sync; sleep 1; sync
	case ${1} in
		mtd6)
			mount -o remount,ro /usr/local
			mount -o remount,rw /usr/local
			;;
		mtd7)
			mount -o remount,ro /home/admin
			mount -o remount,rw /home/admin
			;;
	esac
	sync; sleep 1; sync
}

function softUserappsReset() {
	log "Not enough free space: soft cleaning and retrying installation"
	## cleaning
	for dir in $(ls /home/admin/apps); do
		if [ ! -d fixUpdate ]; then
			rm -rf /home/admin/${dir}
		fi
	done
	for i in $(ls /home/admin/www); do
		if [ -L /home/admin/www/${i} ]; then
			if [[ "${i}" != "shared" ]] || [[ "${i}" != "index.php" ]]; then
				rm -rf /home/admin/www/${i}
			fi
		fi
	done
	## refresh size
	refreshSize "mtd7"
}

function hardUserappsReset() {
	if [ "${1}" = "2" ]; then
		log "Not enough free space: hard cleaning and retrying installation"
	else
		log "ERROR: Installation failed. Blocking RootFS update (ON_BOOT)"
	fi
	## clean
	for file in $(find /home/admin -type f | grep -v "libcrypto\|libssl\|php-cgi"); do
		rm -f ${file}
	done
	## remove VERSION.userappfs to be reinstalled
	mount -o remount,rw /
	rm -f /pkginfo/VERSION.userappfs
	mount -o remount,ro /
	## refresh size
	refreshSize "mtd7"
}

function updateRootfs() {
	#### PLEASE: Do not update the stderr and stdout pipe
	#### Function used to return only 0 and 1

	log "Installing RootFS-Updates." 2>&1 >/dev/null
	source /usr/local/conf/rootfs/VERSION.installer 2>/dev/null 1>/dev/null
	if [ "$(grep $(md5sum ${INSTALLER}) ${PKG_PATH}/conf/rootfs/md5sum.list)" = "" ]; then
		echo 1
		return
	fi
	## run installer
	if ! /bin/cp ${INSTALLER} /tmp 2>/dev/null 1>/dev/null ; then
		echo 1
		return
	fi
	/bin/bash /tmp/rootfs_pkg_install.sh "ON_UPDATE" 2>/dev/null 1>/dev/null

	## remount mtd6 and mtd7 in case pkg installer changed their state
	mount -o remount,rw /usr/local 2>/dev/null 1>/dev/null
	mount -o remount,rw /home/admin 2>/dev/null 1>/dev/null

	if [ -f /tmp/rootfs.issue ]; then
		rm -f /tmp/rootfs.issue
		echo 1
		return
	fi

	echo 0
}

function recursiveInstallFW() {
	if [ "${2}" -ge "3" ]; then
		log "ERROR: firmware can not be installed. Please check your internet connection."
		log "ERROR: If the problem persists, Please contact beroNet support: support@beronet.com"
		touch /tmp/has.issue
		exit 3
	fi
	case $1 in
		clean)
			LEVEL=$((${2} + 1))
			if [ "${LEVEL}" = 1 ] ; then
				softUserappsReset
			else 
				hardUserappsReset "${LEVEL}"
			fi
			recursiveInstallFW "install" "${LEVEL}"
			;;
		install)
			if [ "$(installFW)" = "1" ]; then
				recursiveInstallFW "clean" "${2}"
			else
				recursiveInstallFW "update-rootfs" "${2}"
			fi
			;;
		update-rootfs)
			## move VERSION.* files to /usr/local
			mv /tmp/VERSION /usr/local
			mkdir -p /usr/local/conf/rootfs
			mv /{tmp,home/admin}/conf/rootfs/VERSION.* /usr/local/conf/rootfs
			## install rootfs packages
			if [ "$(updateRootfs)" = "1" ]; then
				if [ "${2}" -le "1" ]; then
					log "ERROR: Updating RootFs failed. Retrying"
				else
					log "Error: Updating RootFs failed. Leaving"
				fi
				recursiveInstallFW "clean" "${2}"
			fi
			## everything goes well. continue installation
			;;
	esac
}

### MAIN ###
[ -f ${LOGFILE} ] && rm ${LOGFILE}

## Remounting AppFs and Userapps space
log "Remounting partitions (rw)."
if ! mount -o remount,rw /usr/local ; then
	log "ERROR: Remounting AppFs-partition (rw) failed"
	touch /tmp/has.issue
	exit 1
fi

if ! mount -o remount,rw /home/admin ; then
	log "ERROR: Remounting UserApps-partition (rw) failed"
	touch /tmp/has.issue
	exit 2
fi

## Installing FW
recursiveInstallFW "install" 0

## cleaning fixUpdate apps (aka, firmware installed from 21.01 or 21.02 or <= 21.03-rc055 using userapps installation mechanism)
if [ -d /home/admin/apps/fixUpdate ]; then
	rm -rf /home/admin/apps/fixUpdate
fi

## Setting permissions
log "Setting permissions"
chown -R root:root /usr/local/*
sync; sleep 1; sync

exit 0
