#!/bin/bash
# VERSION=6
# CHANGES="beroconf DNS works now if 2 dns are configured && updated beronet.com IP in case of no dns"

LOGPATH=/var/log
LOGFILE=${LOGPATH}/system.log
MAXSIZE=1048576

BEROCONF=/usr/fallback/beroconf

# function definitions
function log {

	timestamp=$(date)
	logtype=${1}
	message=${2}

	if [ ! -z "${message}" ]; then
		echo "[${timestamp}] ${logtype}: ${message}" >> ${LOGFILE}
	fi
}

function eval_res {

	if [ ${1} -eq 0 ]; then
		echo "Success"
	else
		echo "Failure"
	fi
}

function log_services {

	service_list="lighttpd sshd isgw beroAvahi watchdog-service netconfigserver"
	process_list=$(/bin/ps)

	for service in ${service_list}; do
		echo "${process_list}" | grep ${service} > /dev/null
		if [ ${?} -eq 0 ]; then
			log "DAEMON" "service ${service} is running"
		else
			log "DAEMON" "service ${service} is not running"
		fi
	done

}

function log_ip {

	bc_dhcp=$(${BEROCONF} get root lan-dhcp | grep -v failed)

	if [[ "${bc_dhcp}" = "yes" ]] || [[ "${bc_dhcp}" = "1" ]]; then
		bc_tmp=$(/sbin/ifconfig eth0 | grep "inet addr:")
		bc_ip=$(expr match "${bc_tmp}" ".*inet addr:\([0-9]\{1,4\}\.[0-9]\{1,4\}\.[0-9]\{1,4\}\.[0-9]\{1,4\}\)")
		bc_nm=$(expr match "${bc_tmp}" ".*Mask:\([0-9]\{1,4\}\.[0-9]\{1,4\}\.[0-9]\{1,4\}\.[0-9]\{1,4\}\)")

		bc_tmp=$(/sbin/route -n | grep "^0.0.0.0")
		bc_gw=$(expr match "${bc_tmp}" ".*[\ ]\{8\}\([0-9]\{1,4\}\.[0-9]\{1,4\}\.[0-9]\{1,4\}\.[0-9]\{1,4\}\)")

		bc_tmp=$(/bin/cat /etc/resolv.conf)
		bc_ns=$(expr match "${bc_tmp}" ".*nameserver \([0-9]\{1,4\}\.[0-9]\{1,4\}\.[0-9]\{1,4\}\.[0-9]\{1,4\}\)")
		unset bc_tmp
	else
		bc_ip=$(${BEROCONF} get root lan-ipaddr | grep -v failed)
		bc_nm=$(${BEROCONF} get root lan-netmask | grep -v failed)
		bc_gw=$(${BEROCONF} get root lan-gateway | grep -v failed)
		bc_ns=$(${BEROCONF} get root lan-nameserver | grep -v failed)
	fi

	log "NETWORK" "Configuration IP: ${bc_ip}/${bc_nm} GW: ${bc_gw} DNS: ${bc_ns}"

	if [ ! -z "${bc_gw}" ]; then
		/bin/ping -c 1 ${bc_gw} > /dev/null
		log "NETWORK" "result of ping to gateway (${bc_gw}): $(eval_res ${?})"
	else
		log "NETWORK" "no gateway configured"
	fi

	if [ ! -z "${bc_ns}" ]; then
		bc_ns_1=$(echo "${bc_ns}" | cut -d ";" -f 1)
		bc_ns_2=$(echo "${bc_ns}" | cut -d -s ";" -f 2)

		if [ ! -z "${bc_ns_1}" ]; then
			/bin/ping -c 1 ${bc_ns_1} > /dev/null
			log "NETWORK" "result of ping to nameserver 1 (${bc_ns_1}): $(eval_res ${?})"
		fi

		if [ ! -z "${bc_ns_2}" ]; then
			/bin/ping -c 1 ${bc_ns_2} > /dev/null
			log "NETWORK" "result of ping to nameserver 2 (${bc_ns_2}): $(eval_res ${?})"
		fi

		/usr/bin/nslookup beronet.com > /dev/null
		log "NETWORK" "result of resolving beronet.com: $(eval_res ${?})"

		/bin/ping -c 1 beronet.com > /dev/null
		log "NETWORK" "result of ping to beronet.com: $(eval_res ${?})"
	else
		log "NETWORK" "no nameserver configured"
		/bin/ping -c 1 35.246.175.191 > /dev/null
		log "NETWORK" "result of ping to 35.246.175.191 (beronet.com): $(eval_res ${?})"
	fi
}

function log_sip {

	sip_list=$(sed -n '/\[.*\]/ {N
			/\naddress/ p}' /usr/conf/isgw.sip)

	IFS=$'\n'

	for s_item in ${sip_list}; do
		if [ ! -z "${s_item}" ]; then
			name=$(expr match "${s_item}" ".*\[\(.*\)\]")
			addr=$(expr match "${s_item}" ".*address = \(.*\)")
			/bin/ping -c 1 ${addr} > /dev/null
			log "SIP" "result of ping to SIP server ${name} (${addr}): $(eval_res ${?})"
		fi
	done

	for r_item in $(/usr/local/sbin/query_isgw r | tr -d '\r'); do
		if [ ! -z "${r_item}" ]; then
			log "SIP" "${r_item}"
		fi
	done
}

function log_port {

	IFS=$'\n'

	for i_item in $(/usr/local/sbin/query_isgw i | tr -d '\r'); do
		if [ ! -z "${i_item}" ]; then
			log "PORT" "${i_item}"
		fi
	done

	for a_item in $(/usr/local/sbin/query_isgw a | tr -d '\r'); do
		if [ ! -z "${a_item}" ]; then
			log "PORT" "${a_item}"
		fi
	done

	for g_item in $(/usr/local/sbin/query_isgw g | tr -d '\r'); do
		if [ ! -z "${g_item}" ]; then
			log "PORT" "${g_item}"
		fi
	done
}

function append_isgw {

	isgw_log=${LOGPATH}/isgw.err

	IFS=$'\n'

	if [ -f ${isgw_log} ]; then
		for i_item in $(cat ${isgw_log}); do
			if [ ! -z "${i_item}" ]; then
				log "ISGW" "${i_item}"
			fi
		done

		rm -rf ${isgw_log}
	fi
}

# run functions

if [ "$(${BEROCONF} get root system-check-enabled | grep -v failed)" != "1" ]; then
	echo "System log is disabled." > ${LOGFILE}
	exit 0
fi

if [ ! -z "$(grep 'System log is disabled.' ${LOGFILE})" ]; then
	rm -f ${LOGFILE}
fi

if [ "${1}" = "init" ]; then
	rm -f ${LOGFILE}
else
	if [ -f ${LOGFILE} ]; then
		cur_size=$(/bin/ls -l ${LOGFILE} | /usr/bin/awk '{ print $5 }')
	fi

	if [ -z "${cur_size}" ]; then
		cur_size=0
	fi

	if [ ${cur_size} -ge ${MAXSIZE} ]; then
		mv ${LOGFILE} ${LOGFILE}.0
	fi
fi

log_services
log_ip
log_sip
log_port
append_isgw

