<?php
	/*
	 * Curl Class to use TLS communication
	 * On TLS errors, use HTTP communication 
	 */
if (!function_exists('curl_init')) {
	## class can not be used
	exit(0);
}

include_once('/usr/php/include/rootHelper.class.php');
class Curl {
	public $bc = null;

	public $has_error = false,
		     $is_init   = false,
		     $error     = null,
				 $log_file  = '';

	public $proxy_data = '';

	public $default_options = array(
		'asJSON'				=> 0,
		'caCrt'         => '',
		'caCrtDefault'  => '/usr/conf/tls/certs/ca-certs.crt', // use default ca-certs (beroCloud). can use a given CA certificate (provisioning)
		'cipherlist'		=> 'TLSv1.2',
		'clientCrt'     => '',
		'clientKey'     => '',
		'connecttimeout'=> 10,
		'debug'					=> 0,
		'handle'				=> null,
		'header'        => null,
		'method'        => 'GET',
		'proxy'         => array(),
		'returntransfer'=> 1,
		'timeout'				=> 10,
		'useragent'     => '',
		'verifyhost'		=> 2,
		'verifypeer'		=> 1,
	);

	/* [CONSTRUCTOR] */
	function __construct($bc = null, $general_options = array()) {
		if (is_null($bc)) {
			require_once('/usr/fallback/beroConf.php');
			$bc = new beroConf('root');
		}
		$this->bc = $bc;
		$this->default_options = array_merge($this->default_options, $general_options);
		$this->is_init = true;
	}

	/* [PUBLIC METHODS] */
	public function hasError() {
		return($this->has_error);
	}

	public function getError() {
		return($this->error);
	}

	public function getSerial() {
		return($this->serial);
	}

	public function log($error) {
		if (strlen($this->log_file)) {
			file_put_contents($this->log_file, rootHelper::getDate() ." $error". PHP_EOL, FILE_APPEND);
			if (filesize($this->log_file) > 3000000) { // aka > 3MB
				copy($this->log_file, "{$this->log_file}.1");
				@unlink($this->log_file);
			}
		}
	}

	public function sendRequest($url, $data = null, $request_options = array(), &$info = array()) {
		if ($this->has_error) {
			return false;
		}
		$options = array_merge($this->default_options, $request_options);

		try {
			if (!$this->is_init) {
				throw new Exception('URL information missing');
			}

			$ch = curl_init();
			if ($ch === false) {
				throw new Exception('Failed to initialize cURL');
			}

			// set basic options
			curl_setopt($ch, CURLOPT_URL, $url);
			curl_setopt($ch, CURLOPT_RETURNTRANSFER, (int) $options['returntransfer']);
			curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, (int) $options['connecttimeout']);
			curl_setopt($ch, CURLOPT_TIMEOUT, (int) $options['timeout']);

			// set header
			if (is_array($options['header'])) {
				curl_setopt($ch, CURL_HTTPHEADER, $options['header']);
			}

			// set useragent
			if (strlen($options['useragent'])) {
				curl_setopt($ch, CURLOPT_USERAGENT, $options['useragent']);
			}

			// TLS management
			if (strlen($this->bc->get('root', 'TLSv1.2-disabled'))) {
				$options['verifyhost'] = 0;
				$options['verifypeer'] = 0;
				$options['cipherlist'] = '';
			}
			else {
				// CA certs
				if (strlen($options['caCrt']) === 0 || !file_exists($options['caCrt'])) {
					$options['caCrt'] = $options['caCrtDefault'];
				}

				$hasCaCrt = false;
				if (file_exists($options['caCrt'])) {
					$hasCaCrt = true;
					curl_setopt($ch, CURLOPT_CAINFO, $options['caCrt']);
				}
				else {
					// a ceritificate is missing. disabling TLS
					$options['verifyhost'] = 0;
					$options['verifypeer'] = 0;
					$options['cipherlist'] = '';
					$this->bc->set('root', 'openssl-ca-missing', 'OPENSSL_CACRT_MISSING');
				}
				if ($hasCaCrt) {
					if (file_exists($options['clientCrt'])) {
						curl_setopt($ch, CURLOPT_SSLCERT, $options['clientCrt']);
					}
					if (file_exists($options['clientKey'])) {
						curl_setopt($ch, CURLOPT_SSLKEY, $options['clientKey']);
					}
					if (strlen($options['cipherlist']) > 0) {
						curl_setopt($ch, CURLOPT_SSL_CIPHER_LIST, $options['cipherlist']);
					}
				}
			}

			curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, (int) $options['verifyhost']);
			curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, (int) $options['verifypeer']);

			// file handling
			if (!is_null($options['handle'])) {
				curl_setopt($ch, CURLOPT_FILE, $options['handle']);
			}

			// method
			switch ($options['method']) {
			case 'DELETE':
			case 'PUT':
				curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $options['method']);
				break;
			case 'GET':
				break;
			case 'POST':
				curl_setopt($ch, CURLOPT_POST, 1);
				break;
			default:
				throw new Exception("Unknown method=$method. Only 'DELETE', 'GET', 'POST' and 'PUT' accepted");
			}

			// proxy
			if (!empty($options['proxy']) && (strlen($options['proxy']['Address']) != 0) && (strlen($options['proxy']['Port']) != 0) && (strlen($options['proxy']['Type']) != 0)) {
				curl_setopt($ch, CURLOPT_PROXY, $options['proxy']['Address']);
				curl_setopt($ch, CURLOPT_PROXYPORT, $options['proxy']['Port']);
				curl_setopt($ch, CURLOPT_PROXYTYPE, $options['proxy']['Type']);
				if (strlen($options['proxy']['User']) != 0 && strlen($options['proxy']['Secret']) != 0) {
					curl_setopt($ch, CURLOPT_PROXYUSERPWD, "{$options['proxy']['User']}:{$options['proxy']['Secret']}");
				}
			}

			// as JSON
			if (!empty($data)) {
				if ($options['asJSON']) {
					curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type:application/json'));
					curl_setopt($ch, CURLOPT_POSTFIELDS, rootHelper::jsonEncode($data));
				}
				else {
					curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
				}
			}

			// debug
			$fp = null;
			if ($options['debug']) {
				curl_setopt($ch, CURLOPT_VERBOSE, 1);
				if ($fp = fopen('/tmp/curl.log', 'w')) {
					curl_setopt($ch, CURLOPT_STDERR, $fp);
				}
			}

			// exec
			$response = curl_exec($ch);
			$info = curl_getinfo($ch);

			// fetch error
			$errno = curl_errno($ch);
			$error = curl_error($ch);

			// close curl
			curl_close($ch);

			if (!is_null($fp)) {
				fclose($fp);
			}

			// debug
			if ($options['debug']) {
				$this->log("###### DEBUG #####\nmethod=$method|uri=$uri|error=$error|errno=$errno|". print_r($data, true).print_r($options, true).print_r($info, true) ."\n###########");
			}

			// manage response
			if (!in_array($info['http_code'], array('200', '201'))) {
				$this->log("HTTP error: [code={$info['http_code']}]");
				return(false);
			}
			else if ($response === false) {
				$this->log("cURL error: [errno=$errno|error=$error]");
				throw new Exception($error, $errno);
			}

			if ($response == false) {
				return(true);
			}
			return($options['asJSON'] ? rootHelper::jsonDecode($response, true) : $response);
		}
		catch (Exception $e) {
			$this->has_error = true;
			$this->error = $this->_setError($e);
			return false;
		}
	}

	/* [PRIVATE METHODS] */
	private function _setError($exception) {
		return array(
			'code'	=> $exception->getCode(),
			'file'	=> $exception->getFile(),
			'line'	=> $exception->getLine(),
			'msg'   => $exception->getMessage(),
		);
	}
}
?>
