#!/usr/bin/php -q
<?php
/* tdmConfigure.php: Script to configure TDM-settings according to hardware.conf
 *
 * Author: Florian Kraatz <fk@beronet.com>
 *
 * Copyright (C) 2016 beroNet GmbH <support@beronet.com>
 *
 *
 * 32 bits are used to configure the tdm
 *
 * Lowest 4 bits [3:0] determine master
 * 0000 - pcm/lif0 
 * 0001 - lif0 is master
 * 0010 - lif1 is master
 * 0011 - lif2 is master
 * 0100 - all lifs are master
 * 
 * lifs are configured in 6 bit steps
 * pcm - [9:4]
 * lif 0 [15:10]
 * lif 1 [21:16]
 * lif 2 [27:22]
 * 
 * meaning of bits
 * bit 0: tausche DR und DX an diesem Interface
 * bit 1: invertiere Shift clock
 * bit 2: invertiere Framesync (L active)
 * bit 3: schalte zweites Framesync an
 * bit 4: betreibe LIF mit langsamerem Takt
 * bit 5: nutze FPGA Takt als Master Takt fuer dieses Interface
 */

require_once('/usr/local/www/berogui/includes/Helper/Helper.php');

/* functions */
function everyLiSelfMaster ($hardwareConf) {
	foreach ($hardwareConf as $sectionName => $section) {
		// ignoring PCM and empty LIs
		if (($sectionName == 'pcm') || (empty($section['type'])) || ($section['type'] == 'none')) {
			continue;
		}

		// if one tdm is slave, the other one is master
		if (!isset($section['master']) || ($section['master'] == 0)) {
			return(false);
		}
	}
	return(true);
}

function checkSipOnly ($hardwareConf) {
	foreach ($hardwareConf as $name => $section) {
		if ($section['master'] == 1) {
			return(false);
		}
	}
	return(true);
}

function isFxModule ($section) {

	if ($section['type'] == 'bf2S02FXS') {
		return(false);
	}

	return(((strstr($section['type'], 'FXS') == false) && (strstr($section['type'], 'FXO') == false)) ? false : true);
}

function isGsmModule ($section) {
	if (strstr($section['type'], 'LTE') != false) {
		return(true);
	}
	return((strstr($section['type'], 'GSM') == false) ? false : true);
}

function whoIsMaster ($hardwareConf) {
	foreach ($hardwareConf as $sectionName => $section) {
		if ($section['master'] == 1) 
			return $sectionName;
	}
}

/* main-program */
if (!file_exists('/usr/conf/hardware.conf')) {
	exit(1);
}

if (Helper::getFpgaVersion() < 7) {
	exit(2);
}

/* reset config file */
//file_put_contents('/sys/class/beronet/pcm/tdm_global', '0');

/* get contents of hardware.conf */
$hardwareConf = parse_ini_file('/usr/conf/hardware.conf', true);

/* init */
$tdm = 0;
$sipOnly == false; 

/* tdm configuration */
$FxConf = array('lif0' => pow(2,15), 
				'lif1' => pow(2,21), 
				'lif2' => pow(2,27));
$GsmConf = array('lif0' => pow(2,13)+pow(2,14)+pow(2,15), 
				 'lif1' => pow(2,19)+pow(2,20)+pow(2,21), 
				 'lif2' => pow(2,25)+pow(2,26)+pow(2,27));
$MasterConf = array('pcm' => pow(2,4), 'lif0' => 1, 'lif1' => 2, 'lif2' => 3, 'all' => 4);
	
/* find master */
if (everyLiSelfMaster($hardwareConf)) {
	$master = 'all';
} else {
	$master = whoIsMaster($hardwareConf);
}

/* set bits for */
$tdm += $MasterConf[$master]; //master

/* FX* lifs */
if ($master == 'all' || $master != 'pcm') {
	foreach ($hardwareConf as $sectionName => $section) {
		if ($sectionName == 'pcm') continue;
		if ( ( $master == 'all' || $master == $sectionName) && isFxModule($section) ) 
			$tdm += $FxConf[$sectionName];
	}
}

/* GSM/LTE lifs */
foreach ($hardwareConf as $sectionName => $section) {
	if ($sectionName == 'pcm') continue;
	if (isGsmModule($section)) 
		$tdm += $GsmConf[$sectionName];
}

/* SIP only */
if (checkSipOnly($hardwareConf)) {
	$tdm = 4+pow(2,15);
	$sipOnly = true;
}

/* write to tdm global config file and logging */
file_put_contents('/sys/class/beronet/pcm/tdm_global', $tdm);

$logData =	"# tdm-bootup.log created on " . Helper::getDate() . "\n" .
			"tdm_config = " . $tdm . "\n" . 
			"tdm_master = " . $master . "\n" .
			"sip_only = " . (($sipOnly == true) ? "yes" : "no") . "\n";

file_put_contents('/var/log/tdm-bootup.log', $logData);

?>