<?php

include_once("/usr/www/include/SQLite2ToSQLite3.php");
include_once '/usr/fallback/beroConf.php';

/**
 * Ehemalige ini_to_db Datei. Ermöglicht das wiederherstellen von Konfigurationsdateien. Die Konfigurationsdateien werden geparst 
 * und die Daten in der Datenkank gespeichert
 * @author Miguel Palmer <mp@beronet.com>
 */
class restoreConfig {

	private $_ba;
	private $_bc;
	private $_log;
	private $_files;
	private $_strpath;
	private $_is_sbc_vm = false;

	/**
	 * @var int Wenn der Wert 0 ist, sind manche Dateien nicht erforderlich damit das Restore erfolgreich ist
	 */
	private $_value;

	function __construct($value = 0, $ba = null) {
		if (!file_exists('/usr/local/www/berogui/misc/database.php')) {
			throw new Exception('missing database.php script');
		}

		include_once("/usr/local/www/berogui/includes/variables.php");
		include_once("/usr/local/www/berogui/misc/database.php");
		$this->_ba = is_null($ba) ? new beroAri() : $ba;
		$this->_is_sbc_vm = $this->_ba->get('is_sbc_vm');
		
		if ($this->_is_sbc_vm) {
			$this->_strpath  = "/etc/beroNet/isgw/";
			$this->_files = array('isgw.conf', 'isgw.dialplan', 'isgw.sip', 'isgw.whitelist', 'misc.conf', 'network.acl');
		}
		else {
			$this->_strpath  = "/usr/conf/";
			$this->_files = array('isgw.conf', 'hardware.conf', 'isgw.isdn', 'isgw.dialplan',
			'isgw.sip', 'misc.conf', 'isgw.causes', 'isgw.analog', 'isgw.cas',
			'isgw.cpt', 'network.acl', 'isgw.gsm', 'isgw.lte', 'isgw.whitelist', 'remotesupport.conf');
		}
		$this->_value = $value;
		
		$this->_bc = new beroConf('root');
	}

	private function error($str, $line) {
		$this->log("FAILURE:" . $str . " in line:" . $line . "\n");
		return false;
	}

	private function log($line) {
		$this->_log .= $line . "\n";
	}

	public function getLog() {
		return $this->_log;
	}

	private function _parseFile($fileName) {
		$this->log("Parsing " . $fileName);
		$fn = explode('.', $fileName);
		$fn = implode("", $fn);
		$result = $this->$fn();
		return $result;
	}

	public function restoreConfig() {
		foreach ($this->_files as $file) {
			$result = $this->_parseFile($file);
			if (!$result)
				return false;
		}
		return true;
	}

	private function _makeParseableInifile($sFile, $dFile) {

		$handle = @fopen($sFile, "r"); // Open file form read.
		$whandle = fopen($dFile, "w");
		if ($handle) {
			while (!feof($handle)) { // Loop til end of file.
				$buffer = fgets($handle, 4096); // Read a line.
				$eqpos = strpos($buffer, "=");
				//Alle Zeilen maskieren da manche Zeilen in dem Value ein = Zeichen enthalten
				if ($eqpos !== false) {
					$key = trim(substr($buffer, 0, $eqpos));
					$value = trim(substr($buffer, $eqpos + 1));
					if (preg_match("/\"/", $value)) {
						$buffer = "$key=$value\n";
					}
					else {
						$buffer = "$key=\"$value\"\n";
					}
				}

				if (strpos($key, "tones") !== false) {
					$buffer = str_replace("[", "", $buffer);
					$buffer = str_replace("]", "", $buffer);
				}
				if (strpos($key, "overlap_dialing") !== false) {
					$buffer = str_replace("yes", "1", $buffer);
					$buffer = str_replace("no", "0", $buffer);
				}
				if (strpos($key, "sip_over_tcp") !== false) {
					$buffer = str_replace("yes", "1", $buffer);
					$buffer = str_replace("no", "0", $buffer);
				}
				if (strpos($key, "time_from_isdn") !== false) {
					$buffer = str_replace("yes", "1", $buffer);
					$buffer = str_replace("no", "0", $buffer);
				}
				if (strpos($key, "qsig") !== false) {
					$buffer = str_replace("yes", "1", $buffer);
					$buffer = str_replace("no", "0", $buffer);
				}
				if (strpos($buffer, "lan-dhcp") !== false) {
					$buffer = str_replace("yes", "1", $buffer);
					$buffer = str_replace("no", "0", $buffer);
				}
				if (strpos($key, "use-boot-prov") !== false) {
					$buffer = str_replace("yes", "1", $buffer);
					$buffer = str_replace("no", "0", $buffer);
				}
				if (strpos($key, "bnfos-hb-on-boot") !== false) {
					$buffer = str_replace("yes", "1", $buffer);
					$buffer = str_replace("no", "0", $buffer);
				}

				$wbuffer = fwrite($whandle, $buffer);
			}
			fclose($handle); // Close the file.
		}
		return;
	}

	private function _saveKeyVal($key, $value, $default = '') {
		if ($value == '') {
			return $this->_ba->set($key, sqlite_escape_string($default));
		} else {
			return $this->_ba->set($key, sqlite_escape_string($value));
		}
	}

	private function _rootDb($key, $value, $check = 0) {
		if ($check == 1 && $value != "") {
			$check = 0;
		}
		if ($check == 0) {
			$this->_bc->set("root", $key, sqlite_escape_string($value));
		}
	}

	private function _insert($table, $keys, $values) {
		// security check
		if (count($keys) != count($values))
		{
			$line = "Error Insert Database; count(key) != count(values); keys = {". implode(",", $keys) ."} and values = {". implode(",", $values) ."}";
			$this->log($line);
			return 1; // database not updated; we return 1 to not corrupt the database
		}

		//Build query
		$structure = 'INSERT INTO ' . $table . ' (';
		foreach ($keys as $key) {
			$structure .= $key . ',';
		}
		$structure = trim($structure, ",");
		$structure .= ') VALUES (';
		foreach ($values as $value) {
			$structure .= '"' . sqlite_escape_string($value) . '",';
		}
		$structure = trim($structure, ",");
		$structure .= ');';
		$this->log($structure);
		return $this->_ba->insert_($structure);
	}

	private function _update($table, $values, $where) {
		//Build query
		$structure = 'UPDATE ' . $table . ' SET ';
		foreach ($values as $key => $value) {
			$structure .= $key . '="' . sqlite_escape_string($value) . '",';
		}
		$structure = trim($structure, ",");
		$structure .= ' ' . $where . ';';
		$this->log($structure);
		return $this->_ba->update($structure);
	}

	
	public function remotesupportconf() {
		
		if (!file_exists($this->_strpath . "remotesupport.conf")) {
			return true;
		}
		
		$kflag=0;
		$keystring = "";
		$cmdstring = "";
		$keycomplete = 0;
		$handle = @file_get_contents("/usr/conf/remotesupport.conf");
		if ($handle) {
			// TODO: optimize it
			$lines = explode("\n", $handle);
			foreach ($lines as $buffer) { // Loop til end of file.
				if (strncmp ($buffer, "SSHCMD:", 7) == 0) {
					$cmdstring = substr ($buffer, 7);
					
				}
				else {
					if (strncmp ($buffer, "-----BEGIN", 10) == 0) {
						$kflag=1;
					}
				}
				if ($kflag) {
					$keystring .= $buffer . "\n";
				}
				if (strncmp ($buffer, "-----END", 8) == 0) {
					if ($kflag) {
						$keycomplete = 1;
					}
					$kflag = 0;
				}
			}
		}
		
		if (strlen($cmdstring) && $keycomplete) {
			foreach (array('permanent_remote_support_active' => 1, 'permanent_remote_support_key' => $keystring, 'permanent_remote_support_cmd', $cmdstring) as $key => $value) {
				$this->_rootDb($key, $value);
			}
		}
		return true;
	}
	
	public function isgwconf() {
		if (!file_exists($this->_strpath . "isgw.conf")) {
			return $this->error("no isgw.conf found", __LINE__);
		}

		$this->_makeParseableInifile($this->_strpath . "isgw.conf", $this->_strpath . "isgw.conf.tmp");

		$result = parse_ini_file($this->_strpath . "isgw.conf.tmp", true);
		//Array $key=>$default
		$fields = array(
			'logserver' => '127.0.0.1:514',
			'syslog_active' => '0',
			'logging_message' => '1',
			'logging_debug' => '0',
			'loglevel_debug' => '0',
			'syslog_facility' => '16',
			'experimental_options' => '',
			'device_name' => '',
			'pcm_server' => '0',
			'pcm_server_port' => '54329',
			'logcdr' => '0',
			'reject_calls_under_load' => '0',
			'emergency_numbers' => '110|112|911|999',
			'blacklist_max_options_packets_lan' => '0',
			'blacklist_options_timeframe_lan' => '0',
			'blacklist_max_invite_packets_lan' => '0',
			'blacklist_invite_timeframe_lan' => '0',
			'blacklist_max_register_packets_lan' => '0',
			'blacklist_register_timeframe_lan' => '0',
			'blacklist_max_other_packets_lan' => '0',
			'blacklist_other_timeframe_lan' => '0',
			'blacklist_blockduration_lan' => '86400',
			'blacklist_max_options_packets_wan' => '0',
			'blacklist_options_timeframe_wan' => '0',
			'blacklist_max_invite_packets_wan' => '0',
			'blacklist_invite_timeframe_wan' => '0',
			'blacklist_max_register_packets_wan' => '0',
			'blacklist_register_timeframe_wan' => '0',
			'blacklist_max_other_packets_wan' => '0',
			'blacklist_other_timeframe_wan' => '0',
			'blacklist_blockduration_wan' => '86400',
			
		);

		foreach ($result as $section => $values) {

			if ($section == "general") {
				$loglevel = $result[$section][loglevel];
				$loglevel_message = $result[$section][loglevel_message];
				//Migration step 3.0
				if ($loglevel_message == "") {
					$loglevel_message = $loglevel;
				}
				if ($loglevel_message == "") {
					$loglevel_message = "2";
				}
				
				$this->_ba->set('loglevel_message', sqlite_escape_string($loglevel_message));
				
				
				foreach ($fields as $key => $default) {
					$res = $this->_saveKeyVal($key, $result[$section][$key], $default);
					if (!$res)
						return $this->error("isgwconf insert failed", __LINE__);
				}
				
				
				//compatibility for update from older dos settings
				$old_timeframe = $result[$section]['blacklist_timeframe'];
				if (strlen ($old_timeframe)) {
						$this->_saveKeyVal('blacklist_options_timeframe_lan', $old_timeframe, 0);
						$this->_saveKeyVal('blacklist_invite_timeframe_lan', $old_timeframe, 0);
						$this->_saveKeyVal('blacklist_register_timeframe_lan', $old_timeframe, 0);
						$this->_saveKeyVal('blacklist_other_timeframe_lan', $old_timeframe, 0);
						$this->_saveKeyVal('blacklist_options_timeframe_wan', $old_timeframe, 0);
						$this->_saveKeyVal('blacklist_invite_timeframe_wan', $old_timeframe, 0);
						$this->_saveKeyVal('blacklist_register_timeframe_wan', $old_timeframe, 0);
						$this->_saveKeyVal('blacklist_other_timeframe_wan', $old_timeframe, 0);
				};
				$old_max_packets = $result[$section]['blacklist_max_packets'];
				if (strlen ($old_max_packets)) {
					$this->_saveKeyVal('blacklist_max_options_packets_lan', $old_max_packets, 0);
					$this->_saveKeyVal('blacklist_max_invite_packets_lan', $old_max_packets, 0);
					$this->_saveKeyVal('blacklist_max_register_packets_lan', $old_max_packets, 0);
					$this->_saveKeyVal('blacklist_max_other_packets_lan', $old_max_packets, 0);
					$this->_saveKeyVal('blacklist_max_options_packets_wan', $old_max_packets, 0);
					$this->_saveKeyVal('blacklist_max_invite_packets_wan', $old_max_packets, 0);
					$this->_saveKeyVal('blacklist_max_register_packets_wan', $old_max_packets, 0);
					$this->_saveKeyVal('blacklist_max_other_packets_wan', $old_max_packets, 0);
				}
				
				$old_blockduration = $result[$section]['blacklist_blockduration'];
				if (strlen ($old_blockduration )) {
					$this->_saveKeyVal('blacklist_blockduration_lan', $old_blockduration, 0);
					$this->_saveKeyVal('blacklist_blockduration_wan', $old_blockduration, 0);
				};
				
			}
		}
		$res = unlink($this->_strpath . "isgw.conf.tmp");
		return true;
	}

	function isgwdialplan() {
	
		if (!file_exists($this->_strpath . "isgw.dialplan")) {
			return $this->error("isgw.dialplan not found", __LINE__);
		}
		$handle = @file_get_contents($this->_strpath . "isgw.dialplan"); // Open file.
		$res = $this->_ba->delete("delete from dialplan");
		if ($handle) {
			$posid = 0;
			$lines = explode("\n", $handle);
			foreach ($lines as $buffer) { // Loop til end of file.
				$ss = 0;
				if ($buffer == ";From\tTo\tFromID\tToID\tDestination\tNewDestination\tSource\tNewSource\tConfig" || strlen($buffer) == 0) {
					continue;
				}
				$result = explode("\t", $buffer);
				if (count($result) > "1") {
					$isdnid = "";
					if (substr($result[4], 0, 10) == "ISDNZUISDN") {
						$isdnid = substr($result[4], 0, 23);
						$result[4] = substr($result[4], 23);
						$ss = 1;
					}
					if (substr($result[5], 0, 10) == "ISDNZUISDN") {
						$isdnid = substr($result[5], 0, 23);
						$result[5] = substr($result[5], 23);
						$ss = 0;
					}

					if (strpos($result[0], ";#") === FALSE) {
						$active = 1;
					} else {
						$result[0] = str_replace(";#", "", $result[0]);
						$active = 0;
					}

					if ($ss == 0) {
						$posid = $posid + 1;
						$keys = array('fromDirection', 'toDirection', 'fromId',
							'toId', 'dest', 'newDest', 'src', 'newSrc', 'config',
							'isdnid', 'posid', 'activ');
						$values = $result;
						if (count($values) == 8) {
							$values[] = "";
						}
						$values[] = $isdnid;
						$values[] = $posid;
						$values[] = $active;
						if (!defined('PHP_VERSION_ID')) {
							$values = str_replace("\\", "\\\\", $values);
						}
						$values = str_replace('""', '\"\"', $values);
						$res = $this->_insert('dialplan', $keys, $values);
					} else {
						$strsql = "update dialplan set 'toID'= '" . sqlite_escape_string($result[3]) . "', toDirection='" . sqlite_escape_string($result[1]) . "' where isdnid='" . sqlite_escape_string($isdnid) . "'";
						$this->log($strsql);
						$res = $this->_ba->insert_($strsql);
					}

					if (!$res)
						return $this->error("isgwdialplan insert failed", __LINE__);
				}
			}
		}
		return true;
	}
	
	function isgwwhitelist() {
	
		if (!file_exists($this->_strpath . "isgw.whitelist")) {
			if ($this->_value == 0) {
				return $this->error("no isgw.whitelist", __LINE__);
			}
			else {
				return true;
			}
		}
		
		$handle = @file_get_contents($this->_strpath . "isgw.whitelist"); // Read file.
		$res = $this->_ba->delete("delete from whitelist");
		if ($handle) {
			$i = 0;
			$lines = explode("\n", $handle);
			foreach ($lines as $buffer) {
				$fields = explode("/", $buffer);
				$field_count = count ($fields);
				if (! ($field_count == 2 || $field_count == 3 )) {
					continue;
				}
				if ($fields[1] < 0 || $fields[1] > 32) {
					continue;
				}
				if ($fields[0] == "") {
					continue;
				}
				
				$sipstack = "ANY";
				if ($field_count == 3) {
					if ($fields[2] == "LAN") {
						$sipstack = "LAN";
					}
					else if ($fields[2] == "WAN") {
						$sipstack = "WAN";
					}
				}
				$strsql = "INSERT INTO whitelist (address, netmask, sipstack) VALUES ('" . sqlite_escape_string($fields[0]) . "','" . sqlite_escape_string($fields[1]) . "','" . sqlite_escape_string($sipstack) . "')";
				$res = $this->_ba->insert_($strsql);
			}
		}
		return true;
	}
	
	function isgwcpt() {
		if (!file_exists($this->_strpath . "isgw.cpt")) {
			if ($this->_value == 0) {
				return $this->error("no isgw.cpt", __LINE__);
			} else {
				return true;
			}
		}
		$handle = @file_get_contents($this->_strpath . "isgw.cpt"); // Read file.
		$res = $this->_ba->delete("delete from callprogress");
		$res = $this->_ba->delete("delete from callprogress_entries");
		if ($handle) {
			$i = 0;
			$lines = explode("\n", $handle);
			foreach ($lines as $buffer) {
				$result = explode("\t", $buffer);

				if (substr($result[0], 0, 1) == "[") {
					$i++;
					$section = substr($result[0], 1, strlen($result[0]) - 2);
					$res = explode(":", $section);

					if ($res[0] == "s2i") {
						$cptDir = 'SIP-ISDN';
					} elseif ($res[0] == "i2s") {
						$cptDir = 'ISDN-SIP';
					} elseif ($res[0] == "s2fxo") {
						$cptDir = 'SIP-FXO';
					}
					$keys = array('id', 'direction', 'name');
					$values = array($i, $cptDir, $res[1]);
					$res = $this->_insert('callprogress', $keys, $values);
					if (!$res) {
						return $this->error("isgw.cpt: tables name insert failed", __LINE__);
					}
				} elseif (substr($result[0], 0, 5) == "TIMER") {
					$res = explode("=", $result[0]);
					$strsql = "UPDATE callprogress set " . sqlite_escape_string(strtolower($res[0])) . "='" . sqlite_escape_string($res[1]) . "' WHERE id='" . $i . "'";
					$res = $this->_ba->update($strsql);
				} elseif (count($result) > 2) {
					$res = null;
					switch ($cptDir) {
						case 'SIP-ISDN':
							$keys = array('type', 'chann_det', 'isdn_event', 'sip_resp', 'resp_text', 'id_name');
							$values = $result;
							$values[] = $i;
							$res = $this->_insert('callprogress_entries', $keys, $values);
							break;
						case 'ISDN-SIP':
							$keys = array('type', 'isdn_event', 'resp_text', 'sip_resp', 'id_name','chann_det');
							$values = $result;
							$values[] = $i;
							$values[]='';
							$res = $this->_insert('callprogress_entries', $keys, $values);
							break;
						case 'SIP-FXO':
							$keys = array('isdn_event', 'sip_resp', 'resp_text', 'id_name','chann_det','type');
							$values = $result;
							$values[] = $i;
							$values[]='';
							$values[]='';
							$res = $this->_insert('callprogress_entries', $keys, $values);
							break;
					}
					if (!$res) {
						return $this->error("isgwcpt: table entries insert failed", __LINE__);
					}
				}
			}
		}
		return true;
	}

	public function isgwsip() {
	
		if (!file_exists($this->_strpath . "isgw.sip")) {
			return $this->error("no isgw.sip found", __LINE__);
		}
		//use old config values as base for LAN stack
		$sipfile = file_get_contents($this->_strpath . "isgw.sip");
		
		$sipfile= preg_replace('/\nsip_address=sip:0.0.0.0:/', "\nsip_bindport_lan=", $sipfile);
		$sipfile= preg_replace('/\nudpports=/', "\nrtpports_lan=", $sipfile);
		$sipfile= preg_replace('/\nsip_transport=/', "\nsip_transport_lan=", $sipfile);
		$sipfile= preg_replace('/\nssl_version=/', "\nssl_version_lan=", $sipfile);
		$sipfile= preg_replace('/\ntos_rtp=/', "\ntos_rtp_lan=", $sipfile);
		$sipfile= preg_replace('/\ntos_sip=/', "\ntos_sip_lan=", $sipfile);
		$sipfile= preg_replace('/\nrtp_port_selection=/', "\nrtp_port_selection_lan=", $sipfile);
		file_put_contents ($this->_strpath . "isgw.sip.tmp2", $sipfile);

		$this->_makeParseableInifile($this->_strpath . "isgw.sip.tmp2", $this->_strpath . "isgw.sip.tmp");
		
		$result = parse_ini_file($this->_strpath . "isgw.sip.tmp", true);
		$res = $this->_ba->delete("delete from sip_general");
		$res = $this->_ba->delete("delete from sip");
		$i = 0;
		foreach ($result as $section => $values) {
			if ($section == "general") {
				$networtInterface = array('lan', 'wan', 'vpn');
				$rtpport=5000;
				foreach ($networtInterface as $interface) {
					if ($interface === 'lan' || isset($result[$section]['sip_bindport_' . $interface])) {
						$fields = array(
							'sip_bindport_' . $interface => '5060',
							'rtpports_' . $interface => ($rtpport+=1000) . '-' . ($rtpport+=1000) , 
							'tos_sip_' . $interface => '160',
							'tos_rtp_' . $interface => '160',
							'rtp_port_selection_' . $interface => '',
							'ssl_version_' . $interface => '',
							'sip_transport_' . $interface => '',
							'stack_active_' . $interface => '1',
							'tls_verification_selection_' . $interface => '',
							"sip_useragent_$interface" => $interface == 'wan' ? SIP_USERAGENT_WAN_DEFAULT : SIP_USERAGENT_LAN_DEFAULT,
							'registration_offset_' . $interface => '0',
								
						);
						$result[$section]["sip_useragent_$interface"] = str_replace(
							array(Helper::getNetworkMacAddr(Helper::getNetworkIfaceByName($interface, $this->_bc)), Helper::getSerial(), implode('_', Helper::getLif())),
							array('{MAC}', '{SERIAL}', '{MODULES}'),
							$result[$section]["sip_useragent_$interface"]
						);
						foreach ($fields as $key => $default) {
							$this->_saveKeyVal($key, $result[$section][$key], $default);
						}
					}
				}
			} else {
				$i++;
				//Sever address was replaced by sip_registrar, sip_domain and sip_proxy
				$address = $result[$section]['address'];
				$sip_registrar = $result[$section]['sip_registrar'];
				$sip_domain = $result[$section]['sip_domain'];
				$sip_proxy = $result[$section]['sip_proxy'];
				//Migration from the old config file
				if ($sip_proxy == "" && $sip_registrar == "" && $sip_domain == "" && $address != "") {
					$result[$section]['sip_proxy'] = $address;
					$result[$section]['sip_registrar'] = $address;
					$result[$section]['sip_domain'] = $address;
				}

				$register = $result[$section]['register'];

				if ($register == "") {
					$register = $result[$section]['type'];
					if ($register == "") {
						$result[$section]['register'] = 0;
					} else {
						if ($register == "both") {
							$result[$section]['register'] = 1;
						} else {
							$result[$section]['register'] = 0;
						}
					}
				}
				//if no validation has been set, try to get it from old register options
				if ($result[$section]['validation'] == "") {
					if (strstr ($result[$section]['register_options'] , "no-validate") ) {
						$result[$section]['validation'] = 0;
					}
					else if (strstr ($result[$section]['register_options'] , "validate") ) {
						$result[$section]['validation'] = 1;
					}
					else {
						$result[$section]['validation'] = 0;
					}
				}
				
				$result[$section]['def'] = $result[$section]['default'];
				$result[$section]['sip_transport'] = $result[$section]['transport'];
				$result[$section]['register_option'] = $result[$section]['register_options'];
				$result[$section]['enum_search_domain'] = $result[$section]['enum_search_domains'];
				$result[$section]['keepalive_interval'] = $result[$section]['keepalive-interval'];

				$result[$section]['name'] = $section;
				$keys = array('name' => '', 'sip_proxy' => '', 'sip_domain' => '', 'sip_registrar' => '',
					'user' => '', 'auth_user' => '', 'displayname' => '', 'sip_transport' => '', 'secret' => '', 'register' => '', 'config' => '', 'def' => '0', 'stunserver' => '',
					'externip' => '', 'register_option' => '', 'regrefresh' => "300", 'regrefresh_failed' => "10", 'matchtype' => '', 'matchtype_manual' => '', 'profile' => 'manual', 'enum_support' => '0', 'enum_search_domain' => '',
					'keepalive_interval' => '0', 'call_transfer_method' => 'refer', 'sip_custom_port' => '', 'sip_stack' => 'lan', 'validation' => '0', 'regrefresh_on_validation_fail' => '0', 'callgroup_name' => '',
					'callgroup_priority' => '', 'max_simultaneous_calls' => '', 'dns_change_registration_behaviour' => '', 'dns_change_minimum_registration_time' => '' );
				$values = array();
				foreach ($keys as $key => $default) {
					if ($result[$section][$key] == '') {
						$values[] = $default;
					} else {
						$values[] = $result[$section][$key];
					}
				}

				$res = $this->_insert('sip', array_keys($keys), $values);

				if (!$res) {
					return $this->error("isgwsip: table sip insert failed", __LINE__);
				}
			}
		}
		$res = unlink($this->_strpath . "isgw.sip.tmp");
		$res = unlink($this->_strpath . "isgw.sip.tmp2");
		return true;
	}

	public function isgwisdn() {

		if (!file_exists($this->_strpath . "isgw.isdn")) {
			return $this->error("no isgw.isdn found", __LINE__);
		}

		$this->_makeParseableInifile($this->_strpath . "isgw.isdn", $this->_strpath . "isgw.isdn.tmp");
		$result = parse_ini_file($this->_strpath . "isgw.isdn.tmp", true);

		foreach ($result as $section => $values) {
			if ($section != "general") {
				$ports = $result[$section]['ports'];
				//Array $colname|$default
				$keys = array('chansel', 'chansel_dir', 'config', 'tones', 'portpullup', 'odt|3',
					'pcmlaw|default', 'country_code', 'city_code', 'local_area_code', 'dialtone_digits',
					'overlap_dialing|0', 'qsig|0', 'odt_empty_dad|15', 'restart_after_link_loss', 'monitored_sip_peer', 'monitored_sip_peer_offtime',
					'gr|' . $section);
				$result[$section]['chansel_dir'] = $result[$section]['chansel_direction'];
				$result[$section]['tones'] = '[' . $result[$section]['tones'] . ']';
				$result[$section]['odt'] = $result[$section]['interdigit_timeout'];
				$result[$section]['odt_empty_dad'] = $result[$section]['interdigit_timeout_initial'];

				$port = explode(",", $ports);

				foreach ($port as $a) {
					$values = array();
					foreach ($keys as $key) {
						$key = explode('|', $key);
						$values[$key[0]] = ($result[$section][$key[0]] == "") ? $key[1] : $result[$section][$key[0]];
					}
					$where = " where port='" . sqlite_escape_string($a) . "'";
					$res = $this->_update('isdn', $values, $where);

					if (!$res)
						return $this->error("isgwisdn: table isdn update failed", __LINE__);
				}
			}else {
				$time_from_isdn = $result[$section][time_from_isdn];
				if ($time_from_isdn == "")
					$time_from_isdn = "0";
				$this->_rootDb("time_from_isdn", $time_from_isdn);

				if ($time_from_isdn != "0")
					$this->_rootDb("ntphost", '');
			}
		}
		@unlink($this->_strpath . "isgw.isdn.tmp");
		return true;
	}

	public function isgwanalog() {

		if (!file_exists($this->_strpath . "isgw.analog")) {
			return $this->error("no isgw.analog found", __LINE__);
		}

		$this->_makeParseableInifile($this->_strpath . "isgw.analog", $this->_strpath . "isgw.analog.tmp");
		$result = parse_ini_file($this->_strpath . "isgw.analog.tmp", true);

		foreach ($result as $section => $values) {
			if ($section != "general") {
				$ports = $result[$section]['ports'];
				$result[$section]['overlap_dial_timeout_empty_dad'] = $result[$section]['interdigit_timeout_initial'];
				$result[$section]['overlap_dial_timeout'] = $result[$section]['interdigit_timeout'];
				$result[$section]['tones'] = '[' . $result[$section]['tones'] . ']';
				$result[$section]['chansel_dir'] = $result[$section]['chansel_direction'];
				$result[$section]['gr'] = $section;
				$port = explode(",", $ports);
				$keys = array('config', 'overlap_dial_timeout_empty_dad', 'overlap_dial_timeout',
					'overlap_dialing', 'fxo_connect', 'fxo_wait_for_oad', 'message_waiting_method', 'tones',
					'clip', 'cnip', 'chansel', 'chansel_dir', 'fxo_dialtone_passthrough', 'fxo_call_ending_signal',
					'fxo_cid_detection_mode', 'fxo_verify_ringing', 'dialtone_digits', 'fxs_wink_offset', 'fxs_wink_duration', 'gr');

				foreach ($port as $a) {
					$values = array();
					foreach ($keys as $key) {
						$values[$key] = $result[$section][$key];
					}
					$where = " where port='" . sqlite_escape_string($a) . "'";
					$res = $this->_update('analog', $values, $where);

					if (!$res)
						return $this->error("isgwanalog: table analog update failed", __LINE__);
				}
			}
		}
		@unlink($this->_strpath . "isgw.analog.tmp");
		// restore custom tones
		if (file_exists('/usr/conf/isgw.tones.custom')) {
			$toinsert = array();
			foreach (parse_ini_file('/usr/conf/isgw.tones.custom', true) as $name => $credentials) {
				$toinsert[] = array_merge(array('name' => $name), $credentials);
			}
			foreach ($toinsert as $tones_custom) {
				if (!$this->_insert('tones_custom', array_keys($tones_custom), $tones_custom)) {
					return $this->error("isgwanalog: tones_custon insert failed", __LINE__);
				}
			}
		}
		return true;
	}

	public function isgwcas() {

		if (!file_exists($this->_strpath . "isgw.cas")) {
			if ($this->_value == 0) {
				return $this->error("no isgw.cas", __LINE__);
			} else {
				return true;
			}
		}

		$this->_makeParseableInifile($this->_strpath . "isgw.cas", $this->_strpath . "isgw.cas.tmp");
		$result = parse_ini_file($this->_strpath . "isgw.cas.tmp", true);

		foreach ($result as $section => $values) {
			if ($section != "general") {
				$ports = $result[$section][ports];
				$result[$section]['e1sigtype'] = $result[$section]['e1_signalling'];
				$result[$section]['t1sigtype'] = $result[$section]['t1_signalling'];
				$result[$section]['tones'] = '[' . $result[$section]['tones'] . ']';
				$result[$section]['chansel_dir'] = $result[$section]['chansel_direction'];
				$result[$section]['gr'] = $section;
				$port = explode(",", $ports);
				$keys = array('config', 'interdigit_timeout_initial', 'interdigit_timeout',
					'overlap_dialing', 'fxo_wait_for_oad', 'tones', 'fxo_callerid_detection_timeout',
					'clip', 'cnip', 'chansel', 'chansel_dir', 'fxo_call_ending_signal', 'pcmlaw',
					'fxo_cid_detection_mode', 'dialtone_digits', 'gr', 'e1sigtype', 't1sigtype', 'wait_for_oad');

				foreach ($port as $a) {
					$values = array();
					foreach ($keys as $key) {
						$values[$key] = $result[$section][$key];
					}
					$where = " where channel='" . sqlite_escape_string($a) . "'";
					$res = $this->_update('cas_channels', $values, $where);

					if (!$res)
						return $this->error("isgwcas: table cas_channels update failed", __LINE__);
				}
			}
		}
		@unlink($this->_strpath . "isgw.cas.tmp");
		return true;
	}

	public function isgwgsm() {

		if (!file_exists($this->_strpath . "isgw.gsm"))
			return $this->error("no isgw.gsm found", __LINE__);

		$this->_makeParseableInifile($this->_strpath . "isgw.gsm", $this->_strpath . "isgw.gsm.tmp");
		$result = parse_ini_file($this->_strpath . "isgw.gsm.tmp", true);

		foreach ($result as $section => $values) {
			if ($section != "general") {
				$ports = $result[$section][ports];
				$result[$section]['sms_extension'] = $result[$section]['sms_extensions'];
				$result[$section]['extension'] = $result[$section]['extensions'];
				$result[$section]['chansel_dir'] = $result[$section]['chansel_direction'];
				$result[$section]['gr'] = $section;
				$port = explode(",", $ports);
				$keys = array('config', 'chansel', 'chansel_dir', 'sms_extension',
					'extension', 'gr');

				$port = explode(",", $ports);
				foreach ($port as $a) {
					$values = array();
					foreach ($keys as $key) {
						$values[$key] = $result[$section][$key];
					}
					$where = " where port='" . sqlite_escape_string($a) . "'";
					$res = $this->_update('gsm', $values, $where);

					if (!$res)
						return $this->error("isgwgsm: table gsm update failed", __LINE__);
				}
			}else {
				$ports = $this->_ba->select("select port from gsm");
				$codec = explode(",", $result[$section]['codec']);
				$pin = explode(",", $result[$section]['pin']);
				$smsc = explode(",", $result[$section]['smsc']);
				while ($entry = $this->_ba->fetch_array($ports)) {
					$key = $entry['port'] - 1;
					$values = array('pin' => $pin[$key], 'smsc' => $smsc[$key],
						'codec' => $codec[$key]);
					$where = " where port='" . $entry['port'] . "'";
					$res = $this->_update('gsm', $values, $where);
				}
			}
		}
		@unlink($this->_strpath . "isgw.gsm.tmp");
		return true;
	}
	
	public function isgwlte() {

		if (!file_exists($this->_strpath . "isgw.lte")) {
			if ($this->_value == 0) {
				return $this->error("no isgw.lte found", __LINE__);
			} else {
				return true;
			}
		}

		$this->_makeParseableInifile($this->_strpath . "isgw.lte", $this->_strpath . "isgw.lte.tmp");
		$result = parse_ini_file($this->_strpath . "isgw.lte.tmp", true);

		foreach ($result as $section => $values) {
			if ($section != "general") {
				$ports = $result[$section][ports];
				$result[$section]['sms_extension'] = $result[$section]['sms_extensions'];
				$result[$section]['extension'] = $result[$section]['extensions'];
				$result[$section]['chansel_dir'] = $result[$section]['chansel_direction'];
				$result[$section]['gr'] = $section;
				$port = explode(",", $ports);
				$keys = array('config', 'chansel', 'chansel_dir', 'sms_extension',
					'extension', 'gr', 'sms_webhook', 'sms_webhook_user', 'sms_webhook_secret', 'sms_webhook_type', 'sms_webhook_id', 'wait_for_cid_timeout', 'lte_init_hook');

				$port = explode(",", $ports);
				foreach ($port as $a) {
					$values = array();
					foreach ($keys as $key) {
						$values[$key] = $result[$section][$key];
					}
					$where = " where port='" . sqlite_escape_string($a) . "'";
					$res = $this->_update('lte', $values, $where);

					if (!$res)
						return $this->error("isgwlte: table lte update failed", __LINE__);
				}
			}
			else {
				$ports = $this->_ba->select("select port from lte");
				$codec = explode(",", $result[$section]['codec']);
				$pin = explode(",", $result[$section]['pin']);
				$smsc = explode(",", $result[$section]['smsc']);
				$provider = explode(",", $result[$section]['provider']);
				while ($entry = $this->_ba->fetch_array($ports)) {
					$key = $entry['port'] - 1;
					$values = array('pin' => $pin[$key], 'smsc' => $smsc[$key], 'codec' => $codec[$key], 'provider' => $provider[$key]);
					$where = " where port='" . $entry['port'] . "'";
					$res = $this->_update('lte', $values, $where);
				}
			}
		}
		@unlink($this->_strpath . "isgw.lte.tmp");
		return true;

	}


	public function isgwcauses() {
		
		$keys = array('direction', 'to_cause', 'from_cause', 'sip_response_message');
		$this->_ba->delete("delete from isgw_causes");
		
		//first we fill the database with the values of the conffs file
		if (file_exists("/usr/local/conf/conffs/isgw.causes")) {
			$causes = @file("/usr/local/conf/conffs/isgw.causes");
			if ($causes != NULL) {
				foreach ($causes as $key => $line) {
					if ($this->_is_sbc_vm && preg_match('/ISDN|ANALOG/', $line)) {
						continue;
					}
					$line = str_replace(array("\r\n", "\n", "\r"), "", $line);
					if ($line != "" && substr_count($line, ";", 0, 1) == 0 && $line != "\r" && $line != "\n" && $line != "\r\n") {
						
						$values = explode("\t", $line);
						if (count($values) == 4 || count($values) == 5 || count ($values) == 6) {
	
							$new_values[0] = $values[0] . "-" . $values[1];
							$new_values[1] = $values[2];
							$new_values[2] = $values[3];
							
							if (count($values) == 4) {
								$new_values[3] = $response_message = "";
							}
							else {
								$new_values[3] = $response_message = $values[4];
							}
							$res = $this->_insert('isgw_causes', $keys, $new_values);
							if (!$res) {// TODO: when error in database, some configurations are lost ... is it possible to avoid this ??
								return $this->error("isgwcauses: table isgw_causes insert failed", __LINE__);
							}
						}
					}
				}
			}
		}
	
		//now we update and extend it with any differences and extensions of the /usr/conf file
		$causes = @file($this->_strpath . "isgw.causes");
		if ($causes != NULL) {
			foreach ($causes as $key => $line) {
				if ($this->_is_sbc_vm && preg_match('/ISDN|ANALOG/', $line)) {
					continue;
				}
				$line = str_replace(array("\r\n", "\n", "\r"), "", $line);
				if ($line != "" && substr_count($line, ";", 0, 1) == 0 && $line != "\r" && $line != "\n" && $line != "\r\n") {
					$values = explode("\t", $line);
					if (count($values) == 4 || count($values) == 5 || count ($values) == 6) {
						$new_values = array();
	
						$direction = $new_values[0] = $values[0] . "-" . $values[1];
						$in_cause = $new_values[1] = $values[2];
						$out_cause = $new_values[2] = $values[3];
						
						if (count($values) == 4) {
							$sip_response = $new_values[3] = "";
						}
						else {
							$sip_response = $new_values[3] = $values[4];
						}

						$query = $this->_ba->select('SELECT * FROM isgw_causes where direction="' . $direction . '" and to_cause="' . $in_cause . '"');
						$record_count = sqlite_num_rows($query);
						if ($record_count == 0) {
							$res = $this->_insert('isgw_causes', $keys, $new_values);
							if (!$res) {// TODO: when error in database, some configurations are lost ... is it possible to avoid this ??
								return $this->error("isgwcauses: table isgw_causes insert failed", __LINE__);
							}
						}
						else if ($record_count == 1) {
							$update_values = array('from_cause' => $out_cause, 'sip_response_message' => $sip_response);
							$where = ' where direction="' . $direction . '" and to_cause="' . $in_cause . '"';
							$res = $this->_update('isgw_causes', $update_values, $where);
						}
					}
				}
			}
		}
		return true;
	}

	public function networkacl() {

		if (!file_exists($this->_strpath . "network.acl")) {
			if ($this->_value == 0)
				return$this->error("no network.acl", __LINE__);
			else
				return true;
		}

		$acl = @file($this->_strpath . "network.acl");
		if ($acl != NULL) {
			$this->_ba->delete("Delete from acl");

			foreach ($acl as $key => $line) {
				if ($line != "" && $line != "\r" && $line != "\n" && $line != "\r\n") {
					$line = str_replace(array("\r\n", "\n", "\r"), "", $line);
					$values = explode(";", $line);
					$address = explode("/", $values[3]);
					
					//ignore invalid lines with no port defined
					if ($values[2] == "") {
						continue;
					}
					if ($values[4] == "DROP") {
						$action = "DROP";
					}
					else {
						$action = "ACCEPT";
					}

					$strsql = "INSERT INTO acl (protocol, dest_port, address, netmask, stack, action) VALUES ('" . sqlite_escape_string($values[1]) . "','" . sqlite_escape_string($values[2]) . "','" . sqlite_escape_string($address[0]). "','" . sqlite_escape_string($address[1]) . "', '" . sqlite_escape_string($stack) . "', '" . sqlite_escape_string($action) . "')";
					if ($DEBUG)
						$this->log($strsql . " \n");
					$res = $this->_ba->insert_($strsql);
				}
			}
		}
		return true;
	}

	function miscconf() {
		if (!file_exists($this->_strpath . "misc.conf"))
			return $this->error("no misc.conf", __LINE__);
		$this->_makeParseableInifile($this->_strpath . "misc.conf", $this->_strpath . "misc.conf.tmp");
		$misc = parse_ini_file($this->_strpath . "misc.conf.tmp", true);

		foreach ($misc as $section => $values) {
			//Time Settings
			$this->_rootDb("timezone", $misc[$section]['timezone']);
			$this->_rootDb("timezone_mode", $misc[$section]['timezone_mode']);
			$this->_rootDb("time_mode", $misc[$section]['time_mode']);
			
			$this->_saveKeyVal("summer_winter_time", $misc[$section]['summer_winter_time'], '');

			//if ntphost host has not been explicitly set, set ntphost to pool.ntp.org
			$gui_ntp = $misc[$section]['ntphost'];
			if (strlen($gui_ntp)) {
				$this->_rootDb("ntphost", $gui_ntp);
				$this->_rootDb("gui-ntphost", $gui_ntp);
			} else {
				$this->_rootDb("ntphost", "pool.ntp.org");
				$this->_rootDb("gui-ntphost", "");
			}
			
			//berofos settings
			$this->_saveKeyVal("ip_berofos", $misc[$section]['bnfos-ipaddr'], '');
			$this->_saveKeyVal("mac_berofos", $misc[$section]['bnfos-mac'], '');
			$this->_saveKeyVal("interval_berofos", $misc[$section]['bnfos-intv'], '');
			$this->_rootDb("use_boot_bnfoswdog", $misc[$section]['bnfos-hb-on-boot'], 1);

			//network settings
			$this->_rootDb("lan-iface", $misc[$section]['lan-iface'], 1);
			$network = require('/usr/local/www/berogui/modules/NetConf/netconf_parameters.php');
			foreach ($network as $param) {
				if (!is_null($param['miscConf'])) {
					$this->_rootDb($param['db-key'], $misc[$section][$param['db-key']], $param['miscConf']);
				}
			}

			foreach (array(
				//MSP Settings
				'msp-iface' => 1, 'msp-ipaddr' => 1,
				//Provisioning
				'use-boot-prov' => 1, 'tftp-host' => 0, 'tftp-url' => 0, 'provisioning_manual' => 1, 'provisioning_useragent' => 1, 'provisioning_mode' => 1, 'provisioning_url' => 1, 'polling_interval' => 1,
			) as $key => $check) {
				$this->_rootDb($key, $misc[$section][$key], $check);
			}

			$this->_rootDb("userapp-enabled", $misc[$section]['userapp-enabled']);

			//Logging
			$this->_rootDb("system-check-enabled", $misc[$section]['system-check-enabled']);

			//Security
			$this->_rootDb("safeguard", $misc[$section]['safeguard']);
			$this->_rootDb("excludelan", $misc[$section]['excludelan']);

			//user
			$adminpass = $misc[$section]['admin-pass'];
			$adminuser = $misc[$section]['admin-user'];
			if ($adminpass != "") {
				$values = array('password' => $adminpass);
				$where = " where user='" . sqlite_escape_string($adminuser) . "'";
				$this->_update('users', $values, $where);
			}
			//misc
			$this->_saveKeyVal("national_code", $misc[$section]['national_code'], '');
			$this->_saveKeyVal("international_code", $misc[$section]['international_code'], '');
			$this->_saveKeyVal("show_app", $misc[$section]['sessions-disable'], '');
			$this->_saveKeyVal("set_hw", $misc[$section]['set_hw'], '');
			$this->_saveKeyVal("lang", $misc[$section]['lang'], '');
			$this->_saveKeyVal("gui_mode", $misc[$section]['gui_mode'], 'simple');


			//API Password
			$apipass = $misc[$section]['api-pass'];
			if (strlen($apipass) != 0) {
				require_once('/usr/local/php/include/apiusers.Class.php');
				$apiusers = new apiusers($apipass);
				$apiusers->createFile();
				$this->log("MISC.CONF:api-pass=" . $apipass . "\n");
			}

			//Cloud / openssl Settings
			foreach (array(
				"cloud_enable", "cloud_key", 'cloud_show_status', 'cloudAddress', 'proxyEnabled', 'proxyAddress', 'proxyPort', 'proxyType', 'proxyUser', 'proxySecret',
			 	'openssl_debug', 'openssl_verifyhost', 'openssl_verifypeer',
			) as $key) {
				$this->_rootDb($key, $misc[$section][$key]);
			}

			//HTTPS
			$this->_saveKeyVal("httpsOnly", $misc[$section]['httpsOnly'], '');

			//SNMP
			//since we made errors in older versions, not exporting to misc.conf, we only put stuff in db which is in misc.conf
			$this->_rootDb("snmp-enabled", $misc[$section]['snmp-enabled']);
			
			//compatibility to old versions, enable server when it has no value but snmp was enabled
			if (!isset($misc[$section]['snmpserver_enabled'])) {
				if ($misc[$section]['snmp-enabled'] == 1) {
					$this->_rootDb("snmpserver_enabled", 1);
					
					if (!isset($misc[$section]['snmp-version'])) {
						$snmp_version = $this->_bc->get("root", "snmp-version");
						if (strlen ($snmp_version) == 0) {
							$this->_rootDb("snmp-version", "SNMPv2");
						}
					}
				}
			}
			else {
				$this->_rootDb("snmpserver_enabled", $misc[$section]['snmpserver_enabled']);
			}
			
			$snmpkeys = array ("snmp-syslocation", "snmp-version", "snmp-community-ro", "snmp-community-rw",
					"snmp-ro-user", "snmp-security-level", "snmp-auth-method", "snmp-encrypt-method", "snmp-ro-auth-password", "snmp-ro-encrypt-password",
					"snmptraps_enabled", "snmptraps_server", "snmptraps_interval", "snmptraps_community",
					"snmptraps_security_level", "snmptraps_auth_method", "snmptraps_encrypt_method", "snmptraps_auth_password", "snmptraps_encrypt_password", "snmptraps_user",
					"snmptraps_traplist", "snmptraps_isdn_crc", "snmptraps_memory_threshold", "snmptraps_cpu_load");
					
			//only write values to db which have been set in misc.conf, otherwise leave the old value in db
			foreach ($snmpkeys as $snmpkey) {
				if (isset ($misc[$section][$snmpkey])) {
					$this->_rootDb($snmpkey, $misc[$section][$snmpkey]);
				}
			}
		}
		$res = unlink($this->_strpath . "misc.conf.tmp");
		return true;
	}

	function hardwareconf() {

		$this->_makeParseableInifile($this->_strpath . "hardware.conf", $this->_strpath . "hardware.conf.tmp");

		$res = $this->_ba->delete("delete from isdn");
		$res = $this->_ba->delete("delete from analog");
		$res = $this->_ba->delete("delete from gsm");
		$res = $this->_ba->delete("delete from lte");
		$res = $this->_ba->delete("delete from cas");
		$res = $this->_ba->delete("delete from cas_channels");
		$cfg = parse_ini_file($this->_strpath . "hardware.conf.tmp", true);

		static $lifport_isdn = 0;
		static $lifport_analog = 0;
		static $lifport_gsm = 0;
		static $lifport_lte = 0;
		static $lifport_cas = 0;

		$master = array();
		foreach ($cfg as $section => $values) {
			$res = true;
			switch ($section) {

				case "lif0":
					$master[] = $cfg[$section]['master'];
					$lifnr = 0;
					$loop = 0;
					$res = $this->_liX($lifnr, $cfg, $section);
					break;
				case "lif1":
					$master[] = $cfg[$section]['master'];
					$lifnr = 1;
					$loop = 0;
					$res = $this->_liX($lifnr, $cfg, $section);
					break;
				case "lif2":
					$master[] = $cfg[$section]['master'];
					$lifnr = 2;
					$loop = 0;
					$res = $this->_liX($lifnr, $cfg, $section);
					break;
				case "pcm":
					if ($cfg[$section]['master'] == 1)
						$master['pcm'] = $cfg[$section]['master'];
					$this->log("Nothing to do for section: " . $section . "\n");
					$loop = 1;
					break;
				default:
					$this->log("Unknown section name: " . $section . "\n");
					$loop = 1;
					break;
			}

			if ($master['pcm'] == 1) {
				$this->_ba->set('master', 'pcm');
				unset($master['pcm']);
			} else {
				$all = true;
				foreach ($master as $key => $value) {
					if ($value == 1)
						$this->_ba->set('master', $key);
					elseif ($value != "")
						$all = false;
				}
				if ($all)
					$this->_ba->set('master', 'all');
			}

			if (!$res)
				return $this->error("hardwareconf: hardware insert failed", __LINE__);


			if ($loop)
				continue;
		}//end first for loop

		@unlink($this->_strpath . "hardware.conf.tmp");
		return true;
	}

	private function _liX($lifnr, $cfg, $section) {
		$type = $cfg[$section]['type'];
		if (!in_array($type, array("bf1t1e1", "bf2t1e1", "bf4t1e1", "bf4S0",
				"bf2S0", "bf1S0", "bf1E1", "bf2E1", "bf2S02FXS", "bf4FXS", "bf8FXS",
				"bf4FXO", "bf2FXO", "bf1FXS", "bf2FXS", "bf2GSM", "bf1GSM", "bf1LTE", "bf2LTE"))) {
			$this->log("Unkown module: " . $type . "\n");
			return true;
		}

		$ports = $this->_getports($type);

		$master = $cfg[$section]['master'];

		$ntteSW = explode(",", $cfg[$section]['ntteSW']);

		$crystal_clock = explode(",", $cfg[$section]['crystal_clock']);

		$permanentl1 = explode(",", $cfg[$section]['permanentl1']);

		$crlen = explode(",", $cfg[$section]['crlen']);
	
		$cidlen = explode(",", $cfg[$section]['cidlen']);

		$crc = explode(",", $cfg[$section]['crc']);

		//CLOCK Setting
		$clock = $cfg[$section]['clock'];
		if (in_array($type, array('bf4S0', 'bf1S0', 'bf1E1', 'bf2E1', 'bf2S0', 'bf2S02FXS'))) {
			($clock == "5") ? $this->_ba->set("sync_port_li" . $lifnr, '1') : $this->_ba->set("sync_port_li" . $lifnr, '0');
		}
		if (in_array($type, array('bf1t1e1', 'bf2t1e1', 'bf4t1e1'))) {
			($clock == "255") ? $this->_ba->set("sync_port_li" . $lifnr, '1') : $this->_ba->set("sync_port_li" . $lifnr, '0');
		}
		if (in_array($type, array('bf4S0', 'bf1S0', 'bf1E1', 'bf2E1', 'bf2S0', 'bf2S02FXS', 'bf1t1e1', 'bf2t1e1', 'bf4t1e1'))) {
			($clock === "0" || $clock == "") ? $this->_ba->set("sync_port_auto" . $lifnr, '1') : $this->_ba->set("sync_port_auto" . $lifnr, '0');

			if ($master == 0)
				$clock = "0";
			$clock = $this->_getdefaultValue("CLOCK", $ports, ($clock == "") ? "0" : $clock);
			if (!$clock)
				return $this->error("hardware.conf->Clock failed", __LINE__);
		}
		//END CLOCK Setting
		//PORTSWITCH Setting
		if (in_array($type, array('bf4S0', 'bf2S02FXS', 'bf4FXS', 'bf4FXO')))
			$this->_saveKeyVal("portswitch", $cfg[$section]['portswitch']);
		//END PORTSWITCH Setting
		//COUNTRY Setting
		if (preg_match('/bf[0-9]*FX[SO]/', $type))
			$this->_saveKeyVal("country", $cfg[$section]['country']);
		//END Country setting
		//BOOSTRINGER Setting, Ring oscillator
		if (preg_match('/bf.*FXS/', $type)) {
			$this->_saveKeyVal("boostringer", $cfg[$section]['boostringer']);
			$this->_saveKeyVal("ring_oscillator_on", $cfg[$section]['ring_oscillator_on']);
			$this->_saveKeyVal("ring_oscillator_off", $cfg[$section]['ring_oscillator_off']);
		}
		//END Boostringer setting
		
		//TERMINATION Setting
		if (in_array($type, array('bf4S0', 'bf2S02FXS', 'bf2S0', 'bf1S0', 'bf1E1', 'bf2E1', 'bf1t1e1', 'bf2t1e1', 'bf4t1e1')))
			$termination = $this->_getdefaultValue("TERMINATION", $ports, explode(",", $cfg[$section]['termination']));
		//END Termination setting
		//PTP Setting
		if (in_array($type, array('bf4S0', 'bf2S02FXS', 'bf2S0', 'bf1S0', 'bf1E1', 'bf2E1')))
			$ptp = $this->_getdefaultValue("PTP", $ports, explode(",", $cfg[$section]['ptp']));
		//END PTP Setting

		$porttype = null;
		if (in_array($type, array('bf4S0', 'bf2S02FXS', 'bf2S0', 'bf1S0'))) {
			$ntteHW = explode(",", $cfg[$section]['ntteHW']);
			$porttype = "BRI";
		}

		if (in_array($type, array('bf1E1', 'bf2E1'))) {
			$ptp = $this->_getdefaultValue("PTP", $ports);
			$ntteHW = $ntteSW;
			$porttype = "PRI";
		}

		if (preg_match('/bf[0-9]*FXS/', $type)) {
			$porttype = "FXS";
		}

		if (in_array($type, array('bf4FXO', 'bf2FXO'))) {
			$this->_saveKeyVal("ohs", $cfg[$section]['ohs']);
			$this->_saveKeyVal("ohs2", $cfg[$section]['ohs2']);
			$this->_saveKeyVal("rz", $cfg[$section]['rz']);
			$this->_saveKeyVal("rt", $cfg[$section]['rt']);
			$this->_saveKeyVal("rt2", $cfg[$section]['rt2']);
			$this->_saveKeyVal("dcv", $cfg[$section]['dcv']);
			$this->_saveKeyVal("ilim", $cfg[$section]['ilim']);
			$this->_saveKeyVal("mini", $cfg[$section]['mini']);
			$this->_saveKeyVal("acim", $cfg[$section]['acim']);
			$this->_saveKeyVal("wait4idle", $cfg[$section]['wait4idle']);
			$porttype = "FXO";
		}

		if (in_array($type, array('bf1t1e1', 'bf2t1e1', 'bf4t1e1'))) {
			$protocol = explode(",", $cfg[$section]['protocol']);
			$country = explode(",", $cfg[$section]['country']);
			$sigchan = explode(",", $cfg[$section]['sigchan']);
			$porttype = explode(",", $cfg[$section]['interface_type']);
			$coding = explode(",", $cfg[$section]['coding']);
		}


		if (in_array($type, array('bf2GSM', 'bf1GSM', 'bf1LTE', 'bf2LTE'))) {
			for ($i = 0; $i < $ports; $i++) {
				if (strpos($type,"LTE")){
					$res = $this->_insert('lte', array('type', 'lif', 'port_type'), array($type, $lifnr, 'LTE'));
				} else {
					$res = $this->_insert('gsm', array('type', 'lif', 'port_type'), array($type, $lifnr, 'GSM'));
				}

				if (!$res)
					return $this->error("hardware.conf->GSM insert failed", __LINE__);
			}
		}elseif (in_array($type, array('bf1t1e1', 'bf2t1e1', 'bf4t1e1'))) {
			for ($i = 0; $i < $ports; $i++) {

				$res = $this->_insert('cas', array('type', 'lif', 'port_type', 'clock', 'protocol', 'country', 'sigchan', 'termination', 'coding'), array($type, $lifnr, $porttype[$i], $clock[$i], $protocol[$i], $country[$i], $sigchan[$i], $termination[$i], $coding[$i]));
				if (!$res)
					return $this->error("hardware.conf->CAS insert failed", __LINE__);
				if (strtolower($porttype[$i]) == "cas") {
					if ($protocol[$i] == "t1")
						$channels = 24;
					elseif ($protocol[$i] == "e1")
						$channels = 31;
					$portCas = $this->_ba->rowid();
					for ($j = 1; $j <= $channels; $j++) {
						$res = $this->_insert('cas_channels', array('port'), array($portCas));
						if ($j == 16 && $protocol[$i] == "e1")
							$channel = $this->_ba->rowid();
						if (!$res)
							return $this->error("hardware.conf->CAS insert failed", __LINE__);
					}
					if ($protocol[$i] == "e1")
						$this->_ba->delete("delete from cas_channels where channel='" . $channel . "';");
					//ISDN Eintrag bei hdlc speichern
				}elseif (strtolower($porttype[$i]) == "hdlc") {
					$res = $this->_insert('isdn', array('type', 'lif', 'port_type', 'ccs','crlen','cidlen'), array($type, $lifnr, 'PRI','1','2','2'));
					if (!$res)
						return $this->error("hardware.conf->ISDN insert failed", __LINE__);
				}
			}
		}elseif ($type == "bf2S02FXS") {
			for ($i = 0; $i < 2; $i++) {
				$keys = array('type', 'lif', 'ptp', 'ntteHW', 'ntteSW',
					'lt', 'clock', 'port_type', 'permanentl1', 'crlen','cidlen');
				$values = array($type, $lifnr, $ptp[$i], $ntteHW[$i], $ntteSW[$i], $termination[$i],
					$clock[$i], 'BRI', $permanentl1[$i],($crlen[$i]!='') ? $crlen[$i] : '1', ($cidlen[$i]!='') ? $cidlen[$i] : '1');
				$res = $this->_insert('isdn', $keys, $values);

				if (!$res)
					return $this->error("hardware.conf->ISDN insert failed", __LINE__);
			}
			for ($i = 0; $i < 2; $i++) {

				$keys = array('type', 'lif', 'port_type');
				$values = array($type, $lifnr, 'FXS');
				$res = $this->_insert('analog', $keys, $values);

				if (!$res)
					return $this->error("hardware.conf->Analog insert failed", __LINE__);
			}
		}elseif (preg_match('/bf[0-9]*FX[SO]/', $type)) {
			for ($i = 0; $i < $ports; $i++) {

				$keys = array('type', 'lif', 'port_type');
				$values = array($type, $lifnr, $porttype);
				$res = $this->_insert('analog', $keys, $values);

				if (!$res)
					return $this->error("hardware.conf->Analog insert failed", __LINE__);
			}
		}else {
			for ($i = 0; $i < $ports; $i++) {
				$keys = array('type', 'lif', 'ptp', 'ntteHW', 'ntteSW',
					'lt', 'clock', 'port_type', 'permanentl1', 'crystal_clock', 'crc', 'crlen', 'cidlen');
				$values = array($type, $lifnr, $ptp[$i], $ntteHW[$i], $ntteSW[$i], $termination[$i],
					$clock[$i], $porttype, $permanentl1[$i], $crystal_clock[$i], $crc[$i],
					($crlen[$i]!='') ? $crlen[$i] : ($porttype=='PRI' ? '2' : '1'),
					($cidlen[$i]!='') ? $cidlen[$i] : ($porttype=='PRI' ? '2' : '1'));
				$res = $this->_insert('isdn', $keys, $values);
				if (!$res)
					return $this->error("hardware.conf->ISDN insert failed", __LINE__);
			}
		}
		return true;
	}

	private function _getports($stype) {
		if (preg_match('/bf(?<ports>[0-9]*).*/', $stype, $matched)) {
			return($matched['ports']);
		}
		return(-1);
	}

	function _getdefaultValue($sname, $iports, $val = null) {

		if ($sname == "CLOCK") {

			$arr = array('0', '1', '2', '3', '4', '5', '254', '255');
			if (!in_array($val, $arr, true))
				return $this->error("hardware.conf: clock out of range", __LINE__);
			if ($val == 5 || $val == 255) {
				for ($i = 1; $i <= $iports; $i++) {
					$tmp[$i - 1] = $val;
				}
			} else {

				for ($i = 1; $i <= $iports; $i++) {


					if ($val == $i) {

						$tmp[$i - 1] = "1";
					} else {

						$tmp[$i - 1] = "0";
					}
				}
			}
			return $tmp;
		}

		if ($sname == "TERMINATION" || $sname == "PTP") {
			if (!is_array($val) || count($val) != $iports) {
				for ($i = 0; $i < $iports; $i++)
					$val[$i] = 1;
			}
			return $val;
		}
	}

}
