<?php
if (!defined('_ROOTHELPER_CLASS')) {
	define('_ROOTHELPER_CLASS', true);

	class rootHelper {
		## // GENERAL FUNCTIONS \\ ##
		public static function convertSecondsToDaily($seconds) {
			if ((int)$seconds <= 0) {
				return array('d' => 0, 'h' => 0, 'm' => 0, 's' => 0);
			}
			$num = floatval($seconds);

			$secs = (int)fmod($num, 60);

			$num = (int)($num / 60);
			$mins = $num % 60;

			$num = (int)($num / 60);
			$hours = $num % 24;

			$days = (int)($num / 24);

			return(array('d' => $days, 'h' => $hours, 'm' => $mins, 's' => $secs));
		}

		/*
		 * Some activate are not necessary from the customer side
		 * And can / must be automatised using this function
		 * Works only for config files following a ini structure
		 *    Associative Array to follow:
		 *        $toActivate[<file_name>] = Array_to_be_converted_in_ini_string
		 *    Example:
		 *        $toActivate['misc.conf'] = array('general' => array('key1' => 'val1', 'key2' => 'val2', ...));
		 */
		public static function doActivate($toActivate) {
			foreach ($toActivate as $file => $params) {
				if (!file_exists("/usr/conf/$file")) {
					// file does not exist => forcing a red activate
					return(self::activate(3));
				}
				file_put_contents("/usr/conf/$file", self::buildIniString(self::arrayReplaceRecursive(self::parseIniFile("/usr/conf/$file"), $params)) ."#include /usr/conf/$file.custom\n");
			}
		}

		/**
		 * reboots device via the init.d/rcK script
		 *
		 * @param string $url legacy option, not used?
		 * @param boolean $apicall  sets window timeout, for gui redirect, if set to false
		 * @return void
		 */
		public static function reboot($url = '', $apicall=false) {
			if (!$apicall) {
				print("<script type='text/javascript'>
						window.setTimeout('setRequest()',15000);
					</script>");
				flush();
			}
			exec('/bin/sh /etc/init.d/rcK > /dev/null &');
		}

		## // SYSTEM - GET FUNCTIONS \\ ##
		public static function getAppFsVersion($is_sbc_vm = 0, $bc = null, $ba = null, $isgw = null) {
			// cloud sbc
			if ($is_sbc_vm) {
				// TODO
				return(null);
			}
			// std sbc
			if (file_exists('/usr/local/VERSION')) {
				if (is_null($bc)) {
					$bc = new beroConf('root');
				}
				preg_match('/appfs-(.*).tar.gz/', $bc->get('root', 'app-image'), $matches);
				return($matches[1]);
			}
			return(null);
		}

		public static function getDate() {
			@exec('/bin/date -R', $output);
			return(trim($output[0]));
		}
		
		public static function getLif() {
			$li = array();
			for ($i = 0; $i < self::getLifCount(); $i++) {
				$li[$i] = self::getLifName($i);
			}
			return($li);
		}

		public static function getLifCount() {
			return(trim(@file_get_contents('/sys/class/beronet/gateway/li_count')));
		}

		public static function getLifName($LifIndex) {
			return(file_exists(self::getLifPath() ."li{$LifIndex}/module") ? trim(@file_get_contents(self::getLifPath() ."li{$LifIndex}/module")) : 'none');
		}

		public static function getLifPath() {
			return(is_dir('/usr/conf/lif') ? '/usr/conf/lif/' : '/sys/class/beronet/');
		}

		public static function getNetworkGateway($Iface = 'eth0', $is_sbc_vm = 0) {
			return(self::exec($is_sbc_vm, '/usr/bin/expr match "$(/sbin/route -n | grep '. $Iface .' | grep UG)" "0.0.0.0 *\([0-9\.]*\)"'));
		}

		public static function getNetworkIfaceByName($name = 'lan', $bc = null) {
			if (is_null($bc)) {
				require_once('/usr/fallback/beroConf.php');
				$bc = new beroConf('root');
			}

			$ports = $bc->get('root', 'lan-ports');
			$mode = $bc->get('root', 'lan-mode');
			$vlan['enable'] = $bc->get('root', 'vlan-enable');
			$vlan['id'] = $bc->get('root', 'vlan-id');

			switch ($name) {
			case 'vpn':
				// lan interface is always 'tun0' if OpenVPN is installed
				if (file_exists('/apps/openvpn/VERSION')) {
					return('tun0');
				}
				break;
			case 'lan':
				switch ($mode) {
				default:
				case 'single':
					// vlan works as of now only on single iface-devices
					if (($vlan['enable'] == 1) && ($vlan['id'] > 0)) {
						return("eth0.{$vlan['id']}");
					}
				case 'lan-wan':
					return('eth0' . (($ports == 2) ? '.10' : ''));
				case 'bonding':
					return('eth0');
				}
				break;
			case 'wan':
				if ($mode == 'lan-wan') {
					return('eth0.11');
				}
				break;
			}
			return(false);
		}

		public static function getNetworkIpAddr($Iface = 'eth0', $is_sbc_vm = 0) {
			$cmd = array(
				0	=> '/usr/bin/expr match "$(/sbin/ifconfig '. $Iface .')" ".*inet addr:\([0-9\.]*\)"',
				1	=> '/usr/bin/expr match "$(/usr/sbin/ifconfig '. $Iface .')" ".*inet \([0-9\.]*\)"',
			);
			return(trim(self::exec($is_sbc_vm, $cmd[$is_sbc_vm])));
		}

		public static function getNetworkMacAddr($Iface = 'eth0') {
			return(trim(@file_get_contents("/sys/class/net/$Iface/address")));
		}

		public static function getNetworkMask($Iface = 'eth0', $is_sbc_vm = 0) {
			$cmd = array(
				0	=> '/usr/bin/expr match "$(/sbin/ifconfig '. $Iface .')" ".* Mask:\([0-9\.]*\)"',
				1 => '/usr/bin/expr match "$(/usr/sbin/ifconfig '. $Iface .')" ".* netmask \([0-9\.]*\)"',
			);
			return(trim(self::exec($is_sbc_vm, $cmd[$is_sbc_vm])));
		}

		public static function getNetworkNameserver() {
			$info = array();
			foreach (file('/etc/resolv.conf') as $line) {
				if (strstr($line, 'nameserver')) {
					$info[] = end(explode(" ", $line));
				}
			}
			return($info);
		}

		public static function getType() {
			return(trim(@file_get_contents('/sys/class/beronet/gateway/type')));
		}

		public static function getResetJumper() {
			return(trim(@file_get_contents('/sys/class/beronet/gateway/resetjumper')));
		}

		## // INFO - GET FUNCTIONS \\ ##
		public static function getFpgaVersion($is_sbc_vm = 0, $bc = null, $ba = null, $isgw = null) {
			// cloud sbc
			if ($is_sbc_vm) {
				return(null);
			}
			// std sbc
			return(trim(@file_get_contents('/sys/class/beronet/fpga/version')));
		}

		public static function getMtdblock($is_sbc_vm = 0, $bc = null, $ba = null, $isgw = null) {
			// cloud sbc 
			if ($is_sbc_vm) {
				return(null);
			}
			// std sbc
			$php_vers_exploded = explode('.', PHP_VERSION);
			return($php_vers_exploded[0] < '7' ? '5' : '7');
		}

		public static function getNetworkInfo($is_sbc_vm = 0, $bc = null, $ba = null, $isgw = null) {
			// cloud sbc
			if ($is_sbc_vm) {
				extract(@parse_ini_file('/etc/beroNet/sbc.conf'));
				$info['wan']['ipaddr'] = $SBCVM_IP;
				$info['wan']['netmask'] = '255.255.255.255';
				if ($bc->get('root', 'is-vpn-running') == 1) {
					$info['lan/vpn']['ipaddr'] = self::getNetworkIpAddr('tun0', 1);
					$info['lan/vpn']['netmask'] = self::getNetworkMask('tun0', 1);
				}
				return($info);
			}
			if (is_null($bc)) {
				require('/usr/fallback/beroConf.php');
				$bc = new beroConf('root');
			}

			// std sbc
			$info['ports'] = $bc->get('root', 'lan-ports');
			$info['mode'] = (strlen($bc->get('root', 'lan-mode')) ? $bc->get('root', 'lan-mode') : 'single');

			// get names of interfaces
			switch ($info['mode']) {
				case 'lan-wan':
					$wan_iface = 'eth0.11';
				case 'single':
					$lan_iface = 'eth0' . (($info['ports'] == 2) ? '.10' : '');
					break;
				case 'bonding':
					$lan_iface = 'eth0';
					break;
			}

			// get lan-info
			$lanGw = false;
			$info['iface']['lan']['macaddr'] = self::getNetworkMacAddr($lan_iface);
			$info['iface']['lan']['ipaddr'] = self::getNetworkIpAddr($lan_iface);
			$info['iface']['lan']['netmask'] = self::getNetworkMask($lan_iface);
			if (strlen($bc->get('root', 'gateway-mode')) == 0 || in_array($bc->get('root', 'gateway-mode'), array('eth0.10', 'manual'))) {
				$info['provided']['gateway'] = self::getNetworkGateway($lan_iface);
				$lanGw = strlen($info['provided']['gateway']) > 0;
			}

			// get wan-info
			if ($info['mode'] == 'lan-wan') {
				$info['iface']['wan']['macaddr'] = self::getNetworkMacAddr($wan_iface);
				$info['iface']['wan']['ipaddr'] = self::getNetworkIpAddr($wan_iface);
				$info['iface']['wan']['netmask'] = self::getNetworkMask($wan_iface);
				if (!$lanGw && in_array($bc->get('root', 'gateway-mode'), array($wan_iface, 'manual'))) {
					$info['provided']['gateway'] = self::getNetworkGateway($wan_iface);
				}
			}

			if ($bc->get('root', 'gateway-mode') == 'manual') {
				$info['provided']['gateway'] = $bc->get('root', 'lan-gateway');
			}

			// get dns-config
			$info['provided']['nameserver'] = self::getNetworkNameserver();

			return($info);
		}

		public static function getOpensslVersion($is_sbc_vm = 0, $bc = null, $ba = null, $isgw = null) {
			$cmd = array(
				0	=> "/usr/local/bin/openssl version",
				1 => "/usr/bin/openssl version",
			);
			$version = self::exec($is_sbc_vm, $cmd[$is_sbc_vm ? 1 : 0]);
			if (preg_match('/OpenSSL ([0-9.a-z]+) .*/', $version, $version_matched)) {
				return($version_matched[1]);
			}	
			// default
			return('1.0.2u (default)');
		}

		public static function getPackageInformation($is_sbc_vm = 1, $bc = null, $ba = null, $isgw = null) {
			if ($is_sbc_vm) {
				extract(@parse_ini_file('/usr/local/VERSION'));
				return array(
					'appfs'		=> $APPFS,
					'beronode'=> $BERONODE,
					'version' => $VERSION,
				);
			}
			return(null);
		}

		public static function getPhpVersion($is_sbc_vm = 0, $bc = null, $ba = null, $isgw = null) {
			return(PHP_VERSION);
		}

		public static function getRevision($is_sbc_vm = 0, $bc = null, $ba = null, $isgw = null) {
			if ($is_sbc_vm) {
				return(null);
			}
			return(trim(@file_get_contents('/sys/class/beronet/gateway/revision')));
		}

		public static function getRootFsVersion($is_sbc_vm = 0, $bc = null, $ba = null, $isgw = null) {
			if ($is_sbc_vm) {
				return(null);
			}
			extract(@parse_ini_file('/pkginfo/VERSION.rootfs'));
			return(trim($PKG_VERSION));
		}

		public static function getSerial($is_sbc_vm = 0, $bc = null, $ba = null, $isgw = null) {
			if ($is_sbc_vm) {
				extract(@parse_ini_file('/etc/beroNet/sbc.conf'));
				return($SBCVM_SERIAL);
			}
			return(trim(@file_get_contents('/sys/class/beronet/gateway/serial')));
		}

		public static function getStorage($is_sbc_vm = 0, $bc = null, $ba = null, $isgw = null) {
			if ($is_sbc_vm) {
				return(null);
			}
			$output = array();
			$storage = array();
			$translate = array(
				5 => 'Root (Rootfs)',
				6 => 'Firmware (Appfs)',
				7 => 'Apps (UserApps)',
				8 => 'Configuration (Conffs)',
			);
			self::exec(0, 'df', $output);
			foreach ($output as $entry) {
				if (preg_match('/^.*mtdblock(?<which>[0-9]+).* (?<used>[0-9]+)%.*$/', $entry, $matched)) {
					$storage[$translate[$matched['which']]] = "{$matched['used']}%";
					unset($matched);
				}
			}
			return($storage);
		}

		static function getUptime($is_sbc_vm = 0, $bc = null, $ba = null, $isgw = null) {
			// cloud sbc
			if ($is_sbc_vm) {
				$time = self::convertSecondsToDaily(@file_get_contents('/proc/uptime'));
				return("{$time['d']}d:{$time['h']}h:{$time['m']}m:{$time['s']}s");
			}
			// std sbc
			$info = array();
			self::exec(0, '/usr/bin/uptime', $resup);
			preg_match("/(?<t>.+), load /", $resup[0], $uptime);
			return($uptime[1]);
		}

		static public function getAppList() {
			$ret = array();
			if (!($dh = @opendir('/home/admin/apps/'))) {
				return $ret;
			}
	
			while (($file = readdir($dh)) !== false) {
				if (($file == '.') || ($file == '..')) {
					continue;
				}
				$ret[] = $file;
			}
			closedir($dh);
	
			return $ret;
		}
		
		## // TLS MANAGEMENT - CLIENT CERTIFICATE \\ ##
		static public function tlsCheckCertificate() {
			if (self::tlsHasCertificate()) {
				## check key and crt match
				$tls = self::tlsGetCertificate();
				$regex = '/^\(stdin\)= (?<md5>.*)$/';
				$keymd5 = exec("/usr/local/bin/openssl rsa -noout -modulus -in {$tls['clientKey']} | /usr/local/bin/openssl md5");
				$crtmd5 = exec("/usr/local/bin/openssl x509 -noout -modulus -in {$tls['clientCrt']} | /usr/local/bin/openssl md5");
				if (preg_match($regex, $keymd5, $keymatched) && preg_match($regex, $crtmd5, $crtmatched) && $keymatched['md5'] === $crtmatched['md5']) {
					## check crt and cacert.pem match
					$verify = exec("/usr/local/bin/openssl verify -verbose -CAfile {$tls['caCrt']} {$tls['clientCrt']}");
					return(preg_match('/^.*: OK$/', $verify) || $verify == 'OK');
				}
			}
			return(false);
		}

		static public function tlsGetCertificate() {
			return(array(
				'caCrt'        => '/usr/conf/tls/cacert.pem',
				'clientCrt'    => '/usr/conf/tls/client.crt',
				'clientKey'    => '/usr/conf/tls/client.key',
			));
		}

		static public function tlsHasCertificate($andSignedByBerocloud = false) {
			$has = true;
			$tls = self::tlsGetCertificate();
			foreach ($tls as $path) {
				$has &= file_exists($path);
			}
			if ($has && $andSignedByBerocloud) {
				$has &= self::tlsIsSignedByBerocloud();
			}
			return($has);
		}

		static public function tlsRenewCertificate($bc = null, &$ptrerror = '') {
			// renew certificate only if managed by beroCloud (aka, serial and mac in CN)
			if (self::tlsIsSignedByBerocloud()) {
				$tls = self::tlsGetCertificate();
				$enddate = strtotime(self::exec(0, "openssl x509 -in {$tls['clientCrt']} -noout -enddate | cut -d '=' -f 2"));
				if ($enddate < (time() + 30*86400)) {
					$startdate = strtotime(self::exec(0, "openssl x509 -in {$tls['clientCrt']} -noout -startdate | cut -d '=' -f 2"));
					$crtLifetime = (int)(($enddate - $startdate)/86400);

					if (is_null($bc)) {
						require_once('/usr/fallback/beroConf.php');
						$bc = new beroconf('root');
					}
					require_once('/usr/local/php/include/berocloud/SbcApi.php');
					$cloud = new SbcApi($bc);
					$res = $cloud->send('renewCertificate', array(
						'lifs' => implode('_', self::getLif($bc)),
						'mac' => self::getNetworkMacAddr(self::getNetworkIfaceByName('lan', $bc)),
						'withTls' => ($crtLifetime > 0 ? $crtLifetime : 3650), // withTls == client certificate lifetime in days
					));
					extract(unserialize($res));
					if (!isset($caCrt) || !isset($clientCrt) || !isset($clientKey) || $error) {
						$ptrerror = 'renewing certificate failed';
						return(false);
					}
					foreach ($tls as $key => $path) {
						file_put_contents($path, $$key);
					}
				}
			}
			return(true);
		}

		static public function tlsIsSignedByBerocloud() {
			$tls = self::tlsGetCertificate();
			return(file_exists($tls['caCrt']) && strpos(self::exec(0, "openssl x509 -in {$tls['caCrt']} -noout -subject"), 'support@beronet.com') !== false);
		}

		## // BERONODE FUNCTIONS \\ ##
		public static function isBeronodeRunning(&$error = null) {
			require_once('/usr/php/include/beronode/ServerApi.php');
			$server_api = new ServerApi();
			$res = $server_api->state();
			if ($res['state'] == 'RUNNING') {
				return(true);
			}
			$error = $server_api->getError();
			return(false);
		}

		## // COMMAND FUNCTIONS \\ ##
		public static function beronodeCommand($cmd) {
			require_once('/usr/php/include/beronode/Api.php');
			$node_api = new Api();
			$response = $node_api->exec($cmd);
			if ($node_api->hasError()) {
				$error = $node_api->getError();
				throw new Exception($error['msg'], $error['code']);
			}
			return(isset($response['success']));
		}

		public static function chown($is_sbc_vm, $owner, $group, $file, $options = array()) {
			if ($is_sbc_vm) {
				try {
					return(self::beronodeCommand("chown ". implode(" ", $options) ." $owner:$group $file"));
				}
				catch (Exception $e) {
					throw $e;
				}
			}
			return(chown($file, $group));
		}

		public static function chmod($is_sbc_vm, $file, $privileges, $options = array()) {
			if ($is_sbc_vm) {
				try {
					return(self::beronodeCommand("chmod $privileges $file"));
				}
				catch (Exception $e) {
					throw $e;
				}
			}
			return(chmod($file, $privileges));
		}

		public static function copy($is_sbc_vm, $source, $dest) {
			if ($is_sbc_vm) {
				try {
					return(self::beronodeCommand("cp '$source' '$dest'"));
				}
				catch (Exception $e) {
					throw $e;
				}
			}
			return(@copy($source, $dest));
		}

		public static function exec($is_sbc_vm, $cmd, &$output = null, $options = array()) {
			if ($is_sbc_vm) {
				require_once('/usr/php/include/beronode/Api.php');
				$node_api = new Api();
				$response = $node_api->exec($cmd, $options);
				// response without error
				if (isset($response['success'])) {
					$output = (isset($response['output']) ? $response['output'] : '');
					return($output);
				}
				// error between berogui / beronode
				else if ($node_api->hasError()) {
					$output = $node_api->getError();
					return(false);
				}
				// command error
				$output = $response['error'];
				return($output);
			}
			$res = exec($cmd, $output);
			return($res);
		}

		public static function jsonDecode($data, $flags = 0) {
			if (PHP_VERSION == '5.2.6') {
				// todo
				return(null);
			}
			return(json_decode($data, $flags));
		}

		public static function jsonEncode($data, $flags = 0) {
			if (PHP_VERSION == '5.2.6') {
				$encoded = '{';
				foreach ($data as $key => $value) {
					if (is_array($value)) {
						$encoded .= "\"$key\":". self::jsonEncode($value) .",";
						continue;
					}
					$encoded .= "\"$key\":\"". trim(str_replace('"', '\"', defined($value) ? constant($value) : $value)) ."\",";
				}
				$encoded = substr($encoded, 0, -1);
				$encoded .= "}";
				return $encoded;
			}
			return json_encode($data, $flags);
		}

		public static function mkdir($is_sbc_vm, $directory, $permissions = 0777, $recursive = false) {
			if ($is_sbc_vm) {
				try {
					$_R = $recursive ? "-p" : "";
					return(self::beronodeCommand("mkdir $_R $directory"));
				}
				catch (Exception $e) {
					throw $e;
				}
			}
			return(mkdir($directory, $permissions, $recursive));
		}

		public static function rename($is_sbc_vm, $inFile, $outFile) {
			if ($is_sbc_vm) {
				try {
					return(self::beronodeCommand("mv '$inFile' '$outFile'"));
				}
				catch (Exception $e) {
					throw $e;
				}
			}
			return(rename($inFile, $outFile));
		}

		public static function rmdir($is_sbc_vm, $dir) {
			if ($is_sbc_vm) {
				try {
					return(self::beronodeCommand("rm -rf '$dir'"));
				}
				catch (Exception $e) {
					throw $e;
				}
			}
			return(rmdir($dir));
		}

		public static function system($is_sbc_vm, $cmd, &$output = null, $options = array()) {
			if ($is_sbc_vm) {
				require_once('/usr/php/include/beronode/Api.php');
				$node_api = new Api();
				$response = $node_api->exec($cmd, $options);
				// response without error
				if (isset($response['success'])) {
					$output = (isset($response['output']) ? $response['output'] : '');
					return(true);
				}
				// error between berogui / beronode
				else if ($node_api->hasError()) {
					$error = $node_api->getError();
					throw new Exception($error['msg'], $error['code']);
				}
				// command error
				$output = $response['error'];
				return(false);
			}
			system($cmd, $retval);
			return($retval);
		}

		public static function unlink($is_sbc_vm, $file) {
			if ($is_sbc_vm) {
				try {
					return(self::beronodeCommand("rm -f '$file'"));
				}
				catch (Exception $e) {
					throw $e;
				}
			}
			return(@unlink($file));
		}

		### ARRAY FUNCTIONS ###
		public static function arrayReplaceRecursive() {
			$args = func_get_args();
			$nbArgs = func_num_args();
			switch ($nbArgs) {
			case 0:
			case 1:
				throw new Exception('at least two arguments required');
			case 2:
				$to_replace = $args[1];
				$to_be_replaced = $args[0];
				if (is_array($to_replace) && is_array($to_be_replaced)) {
					foreach ($to_replace as $key => $values) {
						if (isset($to_be_replaced[$key])) {
							$to_be_replaced[$key] = self::arrayReplaceRecursive($to_be_replaced[$key], $values);
						}
					}
					return($to_be_replaced);
				}
				return($to_replace);
			default:
				$to_be_replaced = $args[0];
				for ($i = $nbArgs; $i >= 2; $i--) {
					$to_be_replaced = self::arrayReplaceRecursive($to_be_replaced, $args[$i-1]);
				}
				return($to_be_replaced);
			}
		}

		### INI FILE FUNCTIONS ###
		public static function buildIniString($data, $str = '', $level = 0) {
			foreach ($data as $key => $value) {
				if (is_array($value) && $level == 0) {
					$str .= "[$key]\n";
					$str = 	self::buildIniString($value, $str, 1);
				}
				else if (is_array($value) && $level == 1) {
					$str .= "$key=". implode(',', $value) ."\n";
				}
				else {
					$str .= "$key=$value\n";
				}
			}
			return("$str\n");
		}

		public static function parseIniFile($file) {
			$parsed = array();
			if (file_exists($file)) {
				$section = 'general';
				foreach (explode("\n", file_get_contents($file)) as $value) {
					if (preg_match('/^\[(.*)\]$/', trim($value), $matched)) {
						$section = trim($matched[1]);
						$parsed[$section] = array();
						unset($matched);
					}
					else if (preg_match('/^(.*)=(.*)(;.*)?$/U', trim($value), $matched)) {
						// manage key. continue when comments
						$key = trim($matched[1]);
						if (strpos($key, ';') !== false) {
							continue;
						}
						// remove "" since not supported by ISGW
						$parsed[$section][$key] = str_replace('"', '', trim($matched[2]));
						unset($matched);
					}
				}
			}
			return($parsed);
		}

		### LOG FUNCTIONS ###
		public static function logRotate($is_sbc_vm, $dir, $file, $opts = array()) {
			// merge options
			$_opts = array_merge(array('nb' => 5), $opts);

			// build and rotate
			file_exists($dir) or self::mkdir($is_sbc_vm, $dir);

			if (isset($_opts['size'])) {
				if (file_exists("$dir/$file") && filesize("$dir/$file") > $_opts['size']) {
					for ($i = ($_opts['nb'] - 1); $i >= 0; $i--) {
						if (file_exists("$dir/$file.$i")) {
							self::rename($is_sbc_vm, "$dir/$file.$i", "$dir/$file.". (string)($i+1));
						}
					}
					self::rename($is_sbc_vm, "$dir/$file", "$dir/$file.0");
				}
			}
			else {
				for ($i = ($_opts['nb'] - 1); $i >= 0; $i--) {
					if (file_exists("$dir/$file.$i")) {
						self::rename($is_sbc_vm, "$dir/$file.$i", "$dir/$file.". (string)($i+1));
					}
				}
				file_exists("$dir/$file") and self::rename($is_sbc_vm, "$dir/$file", "$dir/$file.0");
			}
			return(true);
		}
	}
}
?>
