<?php
	/*
	 * SBC API class to exchange between the SBC and beroCloud
	 */ 
if (!function_exists('curl_init')) {
	## class can not be used
	exit(0);
}

require_once('/usr/php/include/rootHelper.class.php');

class SbcApi {

	public $credentials = array(),
		     $log_file    = '',
				 $useragent   = 'beroNet VoIP Gateway',
				 $url         = '',
				 $version     = "3.0";

	private	$_bc = null,
		      $_ba = null,
					$_ch = null;

	private	$_module = array(
						'analog'=> array('bf1FXS', 'bf2FXS', 'bf4FXS', 'bf8FXS', 'bf2S02FXS', 'bf2FXO', 'bf4FXO'),
						'gsm'		=> array('bf1GSM', 'bf2GSM'),
						'isdn' 	=> array('bf1S0', 'bf2S0', 'bf4S0', 'bf1E1', 'bf2E1', 'bf2S02FXS', 'bf1t1e1', 'bf2t1e1', 'bf4t1e1'),
						'lte'		=> array('bf1LTE', 'bf2LTE'),
					);

	/* [CONSTRUCTOR] */
	function __construct($bc = null, $ba = null, $options = array()) {
		// load database classes
		$this->_bc = $bc;
		if (is_null($this->_bc)) {
			require_once('/usr/fallback/beroConf.php');
			$this->_bc = new beroConf('root');
		}

		$this->_ba = $ba;
		if (is_null($this->_ba) && file_exists('/usr/local/www/berogui/misc/database.php')) {
			require_once('/usr/local/www/berogui/misc/database.php');
			$this->_ba = new beroAri();
		}

		// setup general log
		$logPath = '/usr/conf/permlog/outgoing';
		$logFile = 'berocloud.log';
		rootHelper::logRotate(0, $logPath, $logFile, array('size' => '10000')); // 10000 bytes = 100kb
		$this->log_file = "$logPath/$logFile";

		// load cURL class
		$proxy = array();
		if ($this->_bc->get('root', 'proxyEnabled') == 1) {
			foreach (array('Address', 'Port', 'Secret', 'Type', 'User') as $key) {
				$proxy[$key] = $this->_bc->get('root', "proxy{$key}");
			}
		}
		require_once('/usr/php/include/Curl.class.php');
		$this->_ch = new Curl($this->_bc, array('proxy' => $proxy));

		$logFile = 'berocloud.curl.log';
		rootHelper::logRotate(0, $logPath, $logFile, array('size' => '10000')); // 10000 bytes = 100kb
		$this->_ch->log_file = "$logPath/$logFile";

		// set parameters
		$addr = isset($options['addr']) ? $options['addr'] : $this->_bc->get('root', 'cloudAddress');
		if (strlen($addr) == 0) {
			$addr = 'berocloud.beronet.com';
		}
		$this->url = "https://$addr/index.php";

		// build berocloud credentials
		$this->credentials = array('key' => $this->_bc->get('root', 'cloud_key'), 'serial' => trim(rootHelper::getSerial()));

		$this->is_init = true;
	}

	/* [PUBLIC METHODS] */
	public function getTask() {
		$response = unserialize($this->_send('POST', 'api', array('cmd' => 'getTask')));
		// we log the sane task-requests we get
		if (strlen($response['task']) > 0) {
			$this->log("Received Task '{$response['task']}' from beroCloud");
		}
		switch ($response['task']) {
		case 'activate':
			if ($this->_bc->get('root', 'boot_fwupdate') == 1 || $this->_bc->get('root', 'boot_recoverymode') == 1) {
				break;
			}
			$this->_configActivate();
			break;
		case 'backupConfig':
			$this->_configBackup();
			break;
		case 'coredump':
			$this->_coreDumpUpload();
			break;
		case 'createTunnel':
			$this->log("Creating SSH-Tunnel with {$response['sysCmd']}");
			$this->_createTunnel($response['sshKey'], $response['sysCmd']);
			break;
		case 'deleteCoredump':
			$this->_coreDumpDelete();
			break;
		case 'installApp':
			$this->log("Installing User-App from {$response['file_name']}");
			$handle = @fopen("/tmp/{$response['file_name']}", 'w');
			$this->_send('POST', 'api', array('cmd' => 'appDownload', 'idapp' => $response['idapp']), array('handle' => $handle));
			fclose($handle);
			$this->_updateApp($response['file_name']);
			break;
		case 'Query':
			$this->_queryInfo();
			break;
		case 'reboot':
			$this->_systemReboot();
			break;
		case 'stopSyslog':
			$this->_syslogStop();
			break;
		case 'syslog':
			$this->log("Configuring syslog with IP {$response['ip']} and port {$response['port']}");
			$this->_syslogStart($response['ip'], $response['port']);
			break;
		case 'updateAPI':
			$this->_apiUpdate($response['file']);
			break;
		case 'updateConfig':
			if ($this->_bc->get('root', 'boot_fwupdate') == 1 || $this->_bc->get('root', 'boot_recoverymode') == 1) {
				break;
			}
			$this->log("Updating config from {$response['file_name']}");
			$this->_configUpdate($response['file'], $response['file_name']);
			break;
		case 'updateFW':
			if (strlen($this->_bc->get('root', 'cloudFileName')) == 0) {
				$this->log("Updating Firmware from {$response['file_name']}");
				$this->_bc->set('root', 'cloudFileName', $response['file_name']);
				$this->_bc->set('root', 'boot_fwupdate', 1);
				$this->_configActivate();
				$this->_systemReboot(false);
			}
			break;
		}
		if ($response['next']) {
			$this->getTask();
		}
	}

	public function log($msg) {
		file_put_contents($this->log_file, rootHelper::getDate() ." $msg". PHP_EOL, FILE_APPEND);
	}

	public function renewCa($options = array('connecttimeout' => 2, 'timeout' => 2)) {
		$this->useragent = "beroNet_VoIP:". strtoupper(substr(str_replace(':', '', rootHelper::getNetworkMacAddr(rootHelper::getNetworkIfaceByName('lan', $this->_bc))), 0, 6)) .':renewca';
		if (($newcas = $this->_send('GET', 'api/renewca', array(), $options)) == false) {
			// request failed. ca-file maybe missing. disabling some TLS features
			$newcas = $this->_send('GET', 'api/renewca', array(), array_merge($options, array('verifyhost' => 0, 'verifypeer' => 0)));
		}
		if (strlen($newcas) && $newcas !== 'Invalid Request') {
			$cadefaults = '/usr/conf/tls/certs/ca-defaults.crt';
			if (!file_exists(dirname($cadefaults))) {
				rootHelper::mkdir(0, dirname($cadefaults));
			}
			if (file_put_contents($cadefaults, $newcas)) {
				$cafiles = implode(' ', array_diff(glob('/usr/conf/tls/certs/*'), array('.', '..', '/usr/conf/tls/certs/ca-certs.crt')));
				return(rootHelper::exec(0, "/bin/cat $cafiles > /usr/conf/tls/certs/ca-certs.crt"));
			}
			throw new Exception('Could not write new ca-defaults.crt');
		}
		throw new Exception('Could not download ca-certs from beroCloud');
	}

	public function updateFW($fileName){
		if (strlen($fileName) == 0) {
			$this->log('updateFW:installing firmware failed. FW name missing');
			$this->_bc->delete('root', 'cloudFileName');
			$this->_bc->delete('root', 'boot_fwupdate');
			return;
		}
		require_once('/usr/php/include/updateTool.php');
		$parm = array();
		$parm['userfile']['tmp_name'] ="/tmp/$fileName";
		$parm['userfile']['name'] = $fileName;
		$updateTool = new updateTool();
		$result = $updateTool->upload($parm);

		if ($result['error_code'] == 3) {
			rename("/tmp/$fileName", $updateTool->getImageDir()."/$fileName");
			$updateTool->install($fileName);
			$file_name = $this->_bc->set('root', "cloudFileName", '');
			foreach (array('reboot_after_update', 'hasUpdateFromCloud') as $file) {
				if (file_exists("/tmp/$file")) {
					$this->_systemReboot();
					return(true);
				}
			}
		}
		return(true);
	}

	public function send($action, $data = array(), $options = array(), &$info = array()) {
		$uri = 'api';
		$method = 'POST';
		$toSend = $this->credentials;
		switch ($action) {
		case 'clientTls':
		case 'deregister':
		case 'getAppList':
		case 'register':
			$toSend = array_merge($data, $toSend);
		case 'getAppListCheck':
			$uri = "api/$action";
			$toSend = array_merge($data, $toSend);
			break;
		case 'firmwareDownload':
			$file_name = $this->_bc->get('root', 'cloudFileName');
			if (strlen($file_name) != 0) {
				$toSend = array('cmd' => $action);
				$handle = @fopen("/tmp/$file_name", 'w');
				$res = $this->_send($method, $uri, $toSend, array_merge(array('handle' => $handle), $options), $info);
				fclose($handle);
				if (filesize("/tmp/$file_name") < 1000 && preg_match('/^ERROR=(?<error>.*)$/', file_get_contents("/tmp/$file_name"), $matched)) {
					$this->log("$action:ERROR:{$matched['error']}");
					$this->_bc->delete('root', 'cloudFileName');
					rootHelper::unlink(0, "/tmp/$file_name");
					return(false);
				}
			}
			return($file_name);
		case 'redirect':
			$uri = 'api/redirect';
			$toSend = array_merge($data, $toSend);
			if (strlen($options['redirect_server']) > 0) {
				$uri = '';
				$this->url = $options['redirect_server'];
			}
			break;
		case 'support-access':
			$uri = "api/support-access&key={$data['key']}&serial={$this->credentials['serial']}";
			break;
		default:
			$this->log("Unknown send action request: [action=$action]");
			return false;
		}
		return($this->_send($method, $uri, $toSend, $options, $info));
	}

	public function sendStatus() {
		$isgw = null;
		if (file_exists('/usr/local/www/berogui/includes/isgwtelnet.php')) {
			require_once('/usr/local/www/berogui/includes/isgwtelnet.php');
			$isgw = new isgwtelnet();
			if ($isgw->isgw_login() == false) {
				$isgw = null;
			}
		}
		$data = array('alive' => serialize($this->_aliveData($isgw)));
		if (($status = $this->_statusIsdn($isgw)) !== false) {
			$data['isdnStatus'] = serialize($status);
		}
		if (($status = $this->_statusAnalog($isgw)) !== false) {
			$data['analogStatus'] = serialize($status);
		}
		if (($status = $this->_statusGsm($isgw)) !== false) {
			$data['gsmStatus'] = serialize($status);
		}
		if (($status = $this->_statusLte($isgw)) !== false) {
			$data['lteStatus'] = serialize($status);
		}
		if (($status = $this->_statusSip($isgw)) !== false) {
			$data['sipStatus'] = serialize($status);
		}

		$info = array();
		$this->_send('POST', 'api', $data, array(), $info);
		switch ($info['http_code']) {
		case 200:
		case 201:
			// successful connection. do nothing
			break;
		case 400:
		case 401:
			// update cloud settings only when fully working
			if (strlen($this->_bc->get('root', 'TLSv1.2-disabled')) === 0) {
				// TODO:old code. should we update it ? does it at least happen ??
				sleep(5);
				$this->_bc->set('root', 'cloud_enable', '0');
				$this->_ba->set('error', 'Invalid cloud request. Please check your cloud registration');
				$this->log('Invalid cloud request. Please check your cloud registration - Main request');

				if ($this->_ba->get('activate') <= 1) {
					$this->_ba->set('activate', '1');
				}
				rootHelper::exec(0, '/etc/init.d/S65cloud stop');
			}
			die('Invalid cloud request. Please check your cloud registration');
			break;
		case 418:
			// beroCloud response when unknown serial (aka, not registered)
			if (strlen($this->_bc->get('root', 'TLSv1.2-disabled')) === 0) {
				//// update cloud settings only when fully working
				$this->_bc->set('root', 'cloud_enable', '0');
				$this->_bc->set('root', 'cloud_key', '');
			}
			break;
		default:
			// unknown response
			$this->log("unknown response. sendStatus command. http_code={$info['http_code']}");
			break;
		}
		$this->_ba->set('error', '');
	}

  /* [PRIVATE METHODS - SEND] */
	private function _send($method, $uri, $data, $options = array(), &$info = null) {
		$lighten = array();
		foreach (array('verifypeer', 'verifyhost', 'debug') as $key) {
			$val = $this->_bc->get('root', "openssl_$key");
			if (strlen($val)) {
				$lighten[$key] = $val;
			}
		}
		$options = array_merge($options, $lighten, array('method' => $method, 'useragent' => $this->useragent));
		return($this->_ch->sendRequest(strlen($uri) ? "{$this->url}?r=$uri" : $this->url, array_merge($data, $this->credentials), $options, $info));
	}

	/* [PRIVATE METHODS - GATHERING INFO] */
	private function _aliveData($isgw) {
		// load iswn info
		$isgw_ret['state'] = is_null($isgw) ? 0 : 1;
		if ($isgw_ret['state'] == 1) {
			$isgw_ret['uptime'] = preg_split('/: /', $isgw->isgw_uptime());
			$isgw_ret['actualcalls'] = explode("\n", $isgw->isgw_actualcalls());

			foreach(array('pstn_to_sip', 'sip_to_pstn') as $statName) {
				$statistics[$statName] = file_exists("/tmp/cloud.$statName") ? unserialize(file_get_contents("/tmp/cloud.$statName")) : array();
				// remove old entries from array
				while (count($statistics[$statName]) >= 20) {
					array_shift($statistics[$statName]);
				}
				// get and save data
				foreach ($isgw_ret['actualcalls'] as $statLine) {
					if (preg_match('/.*' . strtoupper($statName) . '.*\((.*) connected\).*/', $statLine, $matches) ) {
						$statistics[$statName][] = $matches[1];
						break; 
					}
				}
				unset($matches);
				file_put_contents("/tmp/cloud.$statName", serialize($statistics[$statName]));
				// calculate average
				$average[$statName] = 0;
				foreach($statistics[$statName] as $value) {
					$average[$statName] += $value;
				}
				$average[$statName] = round($average[$statName] / count($statistics[$statName]));
			}
		}

		// load memory
		$string = file_get_contents('/proc/meminfo');
		preg_match('/.*MemFree:[ ]*([0-9]*)[ ]*kB*/', $string, $free_mem);
		preg_match('/.*Cached:[ ]*([0-9]*)[ ]*kB*/', $string, $cached);

		// coredump
		$coredump = 0;
		$sendCD = 0;
		if (file_exists('/var/log/debug-info.tar.gz')) {
			$coredump = 1;
			clearstatcache();
	  	$time = filemtime('/var/log/debug-info.tar.gz');
			if ($time != $this->_ba->get('time-coredump')) {
				$sendCD = 1;
			}
			file_put_contents('/var/log/time', "$time=". $this->_ba->get('time-coredump') ." $sendCD");
		}

		return(array_merge($this->credentials, array(
			'cmd' 				=> 'alive',
			'coredump' 		=> $coredump,
			'device_name' => $this->_ba->get('device_name'),
			'free_mem' 		=> ($free_mem[1] + $cached[1]),
			'isgw_state' 	=> $isgw_ret['state'],
			'isgw_uptime'	=> $isgw_ret['uptime'][1],
			'load' 				=> strtok(file_get_contents('/proc/loadavg'), ' '),
			'pstn_to_sip'	=> is_null($average['pstn_to_sip']) ? 'null' : $average['pstn_to_sip'],
			'sendCD' 			=> $sendCD,
			'sip_to_pstn' => is_null($average['sip_to_pstn']) ? 'null' : $average['sip_to_pstn'],
			'syslog'			=> file_exists('/var/log/syslog'),
			'uptime' 			=> $uptime['t'],
			'uptime_sec'	=> strtok(file_get_contents('/proc/uptime'), '.'),
		)));
	}

	private function _apiUpdate($file) {
		if (($handle = @fopen('/tmp/cloud-api.php', 'w')) == null) {
			return;
		}
		fwrite($handle, $file);
		fclose($handle);
		rootHelper::system(0, '/bin/mount -o remount,rw /usr/local/');
		rootHelper::system(0, '/bin/cp /tmp/cloud-api.php /usr/local/php/');
		rootHelper::system(0, '/bin/sync');
		rootHelper::system(0, '/bin/mount -o remount,ro /usr/local/');
	}

	private function _configActivate() {
		if (!file_exists('/usr/local/php/create_files.php')) {
			return(false);
		}
		$this->_ba->set('activate', '0');
		rootHelper::exec(0, '/usr/bin/php-cgi /usr/local/php/create_files.php');
		sleep(3);
		rootHelper::system(0, '/usr/local/init/rcK');
		sleep(10);
		$this->_systemReboot(false);
	}

	private function _configBackup() {
		try {
			require_once('/usr/php/include/configExportImportTool.class.php');
			$exp = new configExportImportTool();
			$file = $exp->exportConfig();
			$filename = "beroNetVoIPGateway_{$this->_cardSerial}_". rootHelper::getDate('"+%c"') .'.config.xml';
			file_put_contents("/tmp/$filename", $file);
			$this->_send('POST', 'api', array('cmd' => 'backupConfigResponse', 'file_name' => $filename, 'file' => "@/tmp/$filename"));
			rootHelper::unlink(0, "/tmp/$filename");
		}
		catch (Exception $e) {
			$this->log("_configBackup:{$e->getMessage()}");
		}
	}

	private function _configUpdate($file, $filename = 'beroGui.tar.gz') {
		if ($filename == 'beroGui.tar.gz') {
			if (!file_exists('/usr/local/php/include/confBackup.Class.php')) {
				$this->log('_configUpdate:missing confBackup.Class.php script');
				return;
			}

			try {
				require_once('/usr/local/php/include/confBackup.Class.php');
				$cb = new confBackup();

				$handle = @fopen('/tmp/beroGui.tmp.tar.gz', 'w');
				fwrite($handle, $file);
				fclose($handle);

				if ($cb->installFullBackup ('beroGui.tar.gz', '/tmp/beroGui.tmp.tar.gz') == true) {
					$res = $this->_ba->set('activate','3');
				}
				$cb->installCleanup();
				rootHelper::unlink(0, '/tmp/beroGui.tmp.tar.gz');
				rootHelper::unlink(0, '/tmp/beroGui.tar.gz');
			}
			catch (Exception $e) {
				$this->log("_configUpdate:{$e->getMessage()}");
				return;
			}
		}
		else if (strstr($filename, '.config.xml')) {
			try { 
				if (!file_exists('/usr/local/php/include/restore.Class.php')) {
					$this->log('_configUpdate:missing restore.Class.php script');
					return;
				}

				$handle = @fopen("/tmp/$filename", 'w');
				fwrite($handle, $file);
				fclose($handle);

				require_once('/usr/php/include/configExportImportTool.class.php');
				$imp = new configExportImportTool("/tmp/$filename");
				if (($configFiles = $imp->getConfigFiles()) === false) {
					return;
				}
 
				// create backup-directory
				if (!is_dir('/usr/conf/backup')) {
					rootHelper::mkdir(0, '/usr/conf/backup');
				}
				if (!is_dir('/usr/conf/backup/tls')) {
					rootHelper::mkdir(0, '/usr/conf/backup/tls');
				}
 
				// go over configFiles
				foreach ($configFiles as $configFile) {

					$configName = $configFile['name'];
					$configPath = '/usr/conf';
					$configBackupPath = '/usr/conf/backup';

					switch($configName) {
					case 'caCrt':
						$configName = 'cacert.pem';
						$configPath .= "/tls";
						$configBackupPath .= "/tls";
						break;
					case 'clientCrt':
						$configName = 'client.crt';
						$configPath .= "/tls";
						$configBackupPath .= "/tls";
						break;
					case 'clientKey':
						$configName = 'client.key';
						$configPath .= "/tls";
						$configBackupPath .= "/tls";
						break;
					default:
						break;
					}

					// create backup of file
					@copy("$configPath/$configName", "$configBackupPath/$configName");
					rootHelper::unlink(0, "$configPath/$configName");

					// write file
					if (($FP = fopen("$configPath/$configName", 'w')) == null) {
						@copy("$configBackupPath/$configName", "$configPath/$configName");
						continue;
					}

					fwrite($FP, $configFile['data'], strlen($configFile['data']));
					fclose($FP);
					$FP = null;
				}

				// go over appConfigFiles
				if (($appConfigFiles = $imp->getAppConfigFiles()) !== false) {
					foreach ($appConfigFiles as $app) {
						rootHelper::mkdir(0, "/usr/conf/userapp/{$app['name']}");
						if (empty($app['files'])) {
							continue;
						}
						foreach ($app['files'] as $appFile) {
							switch ($appFile['mode']) {
							case 'directory':
								rootHelper::system(0, "/bin/mkdir -p /usr/conf/userapp/{$appFile['name']}");
								break;
							case 'binary':
								$fileData = base64_decode($appFile['content']);
								break;
							default:
								$fileData = $appFile['content'];
								break;
							}

							if ($appFile['mode'] != 'directory') {
								rootHelper::unlink(0, "/usr/conf/userapp/{$appFile['name']}");
								$FP = fopen("/usr/conf/userapp/{$appFile['name']}", 'w');
								fwrite($FP, $fileData, strlen($fileData));
								fclose($FP);
								$FP = null;
								unset($fileData);
							}

							rootHelper::system(0, "/bin/chown admin:admin /usr/conf/userapp/{$appFile['name']}");
						}
					}
				}

				require_once('/usr/local/php/include/restore.Class.php');
				$restore = new restoreConfig(0);
				$restore->restoreConfig();
				$res = $this->_ba->set('activate','3');
			}
			catch (Exception $e) {
				$this->log("_configUpdate:{$e->getMessage()}");
			}
		}
	}

	private function _coreDumpDelete() {
		foreach (array('/var/log/debug-info.tar.gz', '/var/log/core', '/var/log/corelink') as $file) {
			rootHelper::unlink(0, $file);	
		}
	}

	private function _coreDumpUpload() {
		if (!file_exists('/var/log/debug-info.tar.gz')) {
			return;
		}
		clearstatcache();
		if (($time = filemtime('/var/log/debug-info.tar.gz')) != $this->_ba->get('time-coredump')) {
			$this->_ba->set('time-coredump', $time);
		}
		$file1 = file_exists('/var/log/core') ? '@/var/log/core' : '';
		$this->_send('POST', 'api', array('cmd' => 'coredumpResponse', 'file_name' => 'coredump.tar.gz', 'file' => '@/var/log/debug-info.tar.gz', 'file1' => $file1));
	}

	private function _createTunnel($key, $command) {
		if ((file_put_contents('/tmp/key', $key) === false) || (rootHelper::chmod(0, '/tmp/key', 0600) === false)) {
			return;
		}
		rootHelper::exec(0, "/sbin/$command");
		$this->_send('POST', 'api', array('cmd' => 'createTunnelResponse', 'result' => 'success'));
	}

	private function _queryInfo() {
		$data = array(
			'apiVersion'		=> $this->version,
			'cmd' 					=> 'queryResponse',
			'fpga_version'	=> rootHelper::getFpgaVersion(),
			'hw_rev'				=> rootHelper::getRevision(),
			'ip'						=> rootHelper::getNetworkIpAddr(rootHelper::getNetworkIfaceByName('lan')),
			'lifname'				=> serialize(rootHelper::getLif()), 
			'mac'						=> rootHelper::getNetworkMacAddr(rootHelper::getNetworkIfaceByName('lan')),
			'rootfs_version'=> rootHelper::getRootFsVersion(),
		);
		if (file_exists('/usr/local/www/berogui/includes/Helper/Helper.php')) {
			$data = array_merge($data, array(
				'fw_version'	=> Helper::getAppFsVersion(),
				'apps'				=> serialize(Helper::getAppList()),
			));
		}
		if (file_exists('/usr/local/www/berogui/includes/isgwtelnet.php')) {
			require_once('/usr/local/www/berogui/includes/isgwtelnet.php');
			$isgw = new isgwtelnet();
			if ($isgw->isgw_login()) {
				$data['sip_license_count'] = trim($isgw->isgw_siplicensecount());
				$this->log("sip_license_count:{$data['sip_license_count']}");
			}
		}
		if (!is_null($this->_ba)) {
			$query = $this->_ba->select('SELECT * FROM sip');
			while ($entry = $this->_ba->fetch_array($query)) {
				$sip[] = array('name' => $entry['name'], 'monitor' => $entry['register']);
			}
			$data['sip'] = serialize(!empty($sip) ? $sip : array());
		}
		$this->_send('POST', 'api', $data);
	}

	private function _syslogStart($ip, $port) {
		if (file_exists('/usr/local/www/berogui/includes/isgwtelnet.php')) {
			require_once('/usr/local/www/berogui/includes/isgwtelnet.php');
			$isgw = new isgwtelnet();
			if ($isgw->isgw_login() == false) {
				return;
			}
			file_put_contents('/var/log/syslog', "$ip:$port");
			$isgw->logactive(1);
			$isgw->loglevel(9);
			$isgw->logactive_message(1);
			$isgw->loglevel_message(2);
			$isgw->logserver($ip, $port);
		}
	}

	private function _syslogStop() {
		if (!is_null($this->_ba) && file_exists('/usr/local/www/berogui/includes/isgwtelnet.php')) {
			require_once('/usr/local/www/berogui/includes/isgwtelnet.php');
			$isgw = new isgwtelnet();
			if ($isgw->isgw_login() == false) {
				return;
			}
			$isgw->logactive(0);
			$isgw->loglevel(0);
			$isgw->logactive_message($this->_ba->get('syslog_active'));
			$isgw->loglevel_message($this->_ba->get('loglevel_message'));
			$res = explode(':', $this->_ba->get('logserver'));
			$isgw->logserver($ret[0], $ret[1]);
			rootHelper::unlink(0, '/var/log/syslog');
		}
	}

	private function _systemReboot($fromBerocloud = true) {
		if ($fromBerocloud) {
			## remove recoverymode and fwupdate options from beroconf
			## in case the device is in update / recovery mode
			## and reboot was a task in berocloud
			$this->_bc->delete('root', 'boot_fwupdate');
			$this->_bc->delete('root', 'boot_recoverymode');
		}
		## reboot
		rootHelper::exec(0, '/sbin/reboot');
		exit(0);
	}

	private function _updateApp($fileName) {
		require_once('/usr/php/include/updateTool.php');
		$parm['userfile']['tmp_name'] = "/tmp/{$fileName}";
		$parm['userfile']['name'] = $fileName;
		$upd = new updateTool();
		// Da die updateTool Klasse für Dateien, die durch ein Formular hochgeladen wurden konzipiert
		// wurde, wirft die Klasse einen Fehler, deswegen wird die Datei hier verschoben.
		if ($res['error_code'] == 3) {
			rename("/tmp/$fileName", "{$upd->getImageDir()}/$fileName");
			$upd->install($fileName);
		}
	}

	/* [PRIVATE METHODS - MODULE STATUS] */
	private function _modulesInstalled($moduleNames = null) {
		if ($moduleNames == null) {
			return(false);
		}
		$moduleList = rootHelper::getLif();
		foreach($moduleNames as $moduleName) {
			if (in_array($moduleName, $moduleList)) {
				return(true);
			}
		}
		return(false);
	}

	private function _statusIsdn($isgw) {
		if (!$this->_modulesInstalled($this->_module['isdn']) || is_null($isgw)) {
			return(false);
		}
		// ISDN-Status nochmals einlesen
		$isdnresult = $isgw->isgw_isdnstate();
		$rows = explode("\n", $isdnresult);
		if (file_exists('/proc/l1_stats')) {
			foreach (file('/proc/l1_stats') as $key => $value) {
				foreach (explode(';', $value) as $werte) {
					$wert = explode('=', $werte);
					switch($wert[0]) {
					case 'port':
						$port = $wert[1];
						$l1_stats[$wert[1]] = "";
						break;
					case 'crc':
						$crc[$port] = $wert[1];
						$l1_stats[$port].="CRC Error: " . $wert[1] . "\n";
						break;
					}
				}
			}
		}
		if (file_exists('/proc/l2_stats')) {
			foreach (file('/proc/l2_stats') as $key => $value) {
				foreach (explode(';', $value) as $werte) {
					$wert = explode("=", $werte);
					switch($wert[0]) {
					case 'port':
						$port = $wert[1];
						$l2_stats[$wert[1]] = "";
						break;
					case 'mdl':
						$l2_stats[$port].="L2 Error: " . $wert[1] . "\n";
						break;
					case 'frm':
						$l2_stats[$port].="L2 Frame Error: " . $wert[1] . "\n";
						break;
					}
				}
			}
		}
		$i = 0;
		foreach ($rows as $value) {
			if (strpos($value, 'Port') !== false) {
				preg_match('&\* Port (?<port>\d+) Type (?<type>\w+) Prot. (?<prot>\w+) L2Link (?<l2>\w+) L1Link:(?<l1>\w+) Blocked:0&isU', $value, $arr);

				$isdnStatus[$i]['port'] = $arr['port'];
				$isdnStatus[$i]['state'] = $arr['l2'];
				$isdnStatus[$i]['ISDN CRC Error'] = $crc[$arr['port']];
				$isdnStatus[$i]['ISDN Port State L1'] = ($arr['l1'] == "UP") ? "1" : "0";
				$isdnStatus[$i]['ISDN Port State L2'] = ($arr['l2'] == "UP") ? "1" : "0";
				$isdnStatus[$i]['state'] = $arr['l2'];
				$isdnStatus[$i++]['message'] = 'L2Link:' . $arr['l2'] . "\n" . 'L1Link:' . $arr['l1'] . "\n" . $l1_stats[$arr['port']];
			}
		}

		return(array_merge($this->credentials, array(
			'cmd' 	=> 'isdnStatus',
			'status'=> serialize(!empty($isdnStatus) ? $isdnStatus : array()),
		)));
	}

	private function _statusAnalog($isgw) {
		if (!$this->_modulesInstalled($this->_module['analog']) || is_null($isgw)) {
			return(false);
		}

 		$i = 0;
		$analogresult = $isgw->isgw_analogstate();
		foreach(explode("\n", $analogresult) as $value) {
			if (strpos($value, 'Port') === false) {
				continue;
			}

			$tmp = explode("Port ", $value);
			$port = explode("type:", $tmp[1]);
			$message = "Port: " . $port[0] . "\n";
			$analogStatus[$i]['port'] = $port[0];
			$type = explode("state:", $port[1]);
			$message .= "Type: " . $type[0] . "\n";
			$state = explode("linevoltage:", $type[1]);
			$message .= "State: " . $state[0] . "\n";
			$linevoltage = explode("pol:", $state[1]);
			$message .= "Line voltage: " . $linevoltage[0] . "\n";
			$analogStatus[$i]['state'] = ($linevoltage[0] >= 30) ? 'UP' : 'DOWN';
			$pol = explode("rtlc:", $linevoltage[1]);
			$message .= "Polarity: " . $pol[0] . "\n";
			$analogStatus[$i++]['message'] = $message;
		}

		return(array_merge($this->credentials, array(
			'cmd' 	=> 'analogStatus',
			'status' => serialize(!empty($analogStatus) ? $analogStatus : array()),
		)));
	}

	private function _statusGsm($isgw, $module = 'gsm') {
		if (!$this->_modulesInstalled($this->_module[$module]) || is_null($isgw)) {
			return(false);
		}

		$i = 0;
		$gsmStatus = array();
		foreach(explode("\n", $isgw->isgw_gsmstate()) as $value) {
			if (preg_match('/.*Port(.*)Provider:(.*)Pin Counter:(.*)Reg Status:(.*)RSSI:(.*)BER:(.*)/', $value, $matched)) {
				$status = '';
				$port = trim($matched[1]);
				$rssi = trim($matched[5]);
				$gsmStatus[$i]['port'] = $port;
				$gsmStatus[$i]['state'] = $this->_gsmFetchStatus(trim($matched[4]), $status);
				$gsmStatus[$i++]['message'] = "Port: $port\nProvider: ". trim($matched[2]) ."\nPIN Counter: ". trim($matched[3]) ."\n$status\nSignal: ". $this->_gsmFetchSignal($rssi) ."\n"
																		.	"RSSI: $rssi\nBER: ". trim($matched[6]) ."\n";
				unset($matched);
			}
		}

		return(array_merge($this->credentials, array(
			'cmd' 	=> 'gsmStatus',
			'status'=> serialize(!empty($gsmStatus) ? $gsmStatus : array()),
		)));
	}

	private function _statusLte($isgw) {
		if (($gsm = $this->_statusGsm($isgw, 'lte')) == false) {
			return(false);
		}
		return(array_merge($gsm, array('cmd' => 'lteStatus')));
	}

	private function _statusSip($isgw) {
		if (is_null($isgw)) {
			return(false);
		}
		$i = 0;
		$sipresult = $isgw->isgw_sipstate();
		foreach (explode("\n", $sipresult) as $value){
			preg_match('/(.+)\s*\=\>(.+)\s*/', $value, $match);
			switch (trim($match[1])) {
			case 'ACCOUNTNAME':
				$name = trim($match[2]);
				$sipStatus[$i]['message'] .= preg_replace("/\s+/", "", $value);
				break;
			case 'ADDRESS':
			case 'USERNAME':
				$sipStatus[$i]['message'] .= preg_replace("/\s+/", "", $value);
				break;
			case 'REGSTATUS':
				$sipStatus[$i]['name'] = $name;
				$sipStatus[$i]['state'] = trim($match[2]);
				$sipStatus[$i]['message'] .= preg_replace("/\s+/", "", $value);
				unset($name);
				$i++;
				break;
			}
		}
		return(array_merge($this->credentials, array(
			'cmd' 	=> 'sipStatus',
			'status'=> serialize(!empty($sipStatus) ? $sipStatus : array()),
		)));
	}

	/* [PRIVATE METHODS - GSM] */
	private function _gsmFetchSignal($input) {
		if ($input == -1 || $input == 99) {
			return '-';
		}
		else if ($input > -60) {
			return 'VERY HIGH';
		}
		else if ($input <= -60 && $input > -75) {
			return 'HIGH';
		}
		else if ($input <= -75 && $input > -100) {
			return 'OK';
		}
		else if ($input <= -100) {
			return 'LOW';
		}
		return 'UNKNOWN';
	}

	private function _gsmFetchStatus($input, &$message = null) {
		$message = 'Reg status: ';
		switch ($input) {
		case 'NOT READY':
			$message .= 'not ready';
			return 'DOWN';
		case 0:
			$message .= 'not registered, ME is <b>not</b> currently searching a new operator to register to';
			return 'DOWN';
		case 1:
			$message .= 'registered, home network';
			return 'UP';
		case 2:
			$message .= 'not registered, but ME is currently searching a new operator to register to';
			return 'DOWN';
		case 3:
			$message .= 'registration denied';
			return 'DOWN';
		case 4:
			$message .= 'unknown';
			return 'DOWN';
		case 5:
			$message .= 'registered, roaming';
			return 'UP';
		default:
			$message .= 'unknown status';
			return 'DOWN';
		}
	}
}
?>
