#!/bin/bash

## MAIN VARIABLES ##
BEROCONF=/usr/fallback/beroconf

SETENV=/usr/fallback/fw_setenv
PRINTENV=/usr/fallback/fw_printenv

source /usr/fallback/helper

## EXIT FUNCTION ##
function actOnIssue() {
	## reboot in normal mode
	permlog "reproduce" "[reproduce]" "${1}"
	if [ -f /tmp/reproduce.gui ]; then
		${BEROCONF} set root reproduce-failed REPRODUCE_FAILED
		${BEROCONF} delete root boot_recoverymode
		/sbin/reboot &>/dev/null
	fi
	exit 1	
}

## UPDATE U-BOOT FUNCTION ##
function updateUboot() {
	if ! ${SETENV} ${1} &>/dev/null; then
		actOnIssue ${2}
	fi	
}

## CHECK TFTP binary
TFTP=$(/usr/bin/which tftp)
[ -z ${TFTP} ] && actOnIssue "Missing tftp binary"

## CHECK REQUIRED FILES ##
[ ! -f ${SETENV} ] && actOnIssue "'fw_setenv' file missing"
[ ! -f ${PRINTENV} ] && actOnIssue "'fw_printenv' file missing"

## CHECK INPUT ##
function checkSubnet() {
	localip=($(echo "$1" | tr '.' '\n'))
	remoteip=($(echo "$2" | tr '.' '\n'))
	netmask=($(echo "255.255.255.0" | tr '.' '\n'))

	for i in {0..3}; do
		if [ ! "$((${localip[$i]} & ${netmask[$i]}))" = "$((${remoteip[$i]} & ${netmask[$i]}))" ]; then
			echo 1
			return
		fi
	done
	echo 0
}

LOCALIP=${1}
[ -z ${LOCALIP} ] && actOnIssue "Usage: ${0} [localIp] [remoteIp] [remotePath] [fw]"

REMOTEIP=${2}
[ -z ${REMOTEIP} ] && actOnIssue "Missing Remote IP-Address"

[ "$(checkSubnet "${LOCALIP}" "${REMOTEIP}")" = "1" ] && actOnIssue "localIp and remoteIp not in the same subnet"

REMOTEPATH=${3}
[ -z ${REMOTEPATH} ] && actOnIssue "Missing Remote Path"

## LOAD MD5SUM FILE ##
if [ ! -f /tmp/reproduce.gui ]; then
	## always download md5sum in case the firmware value changed
	[ -f /tmp/reproduce.md5sum ] && /bin/rm -f /tmp/reproduce.md5sum
	## file missing. downloading from beroNet websserver
	FW=${4}
	[ -z ${FW} ] && actOnIssue "Firmware value missing"
	if ! /usr/bin/wget ftp://appliance.beronet.com/beronet_gateway/reproduce/${FW}.md5sum -O /tmp/reproduce.md5sum &>/dev/null ; then
		if ! /usr/bin/wget ftp://85.214.59.198/beronet_gateway/reproduce/${FW}.md5sum -O /tmp/reproduce.md5sum &>/dev/null ; then
			cd /tmp
			if ! /bin/echo "get ${REMOTEPATH}/${FW}.md5sum" | ${TFTP} ${REMOTEIP} &>/dev/null ; then
				actOnIssue "md5sum file missing"
			fi
			/bin/mv ${FW}.md5sum reproduce.md5sum
		fi
	fi
fi
source /tmp/reproduce.md5sum

## CHECK TFTP-CONNECTION ##
if ! /bin/ping -c 1 ${REMOTEIP} &>/dev/null; then
	actOnIssue "TFTP Server unavailable"
elif ! /bin/echo "quit" | ${TFTP} ${REMOTEIP} &>/dev/null; then
	actOnIssue "TFTP binary unstable / not working properly"
fi

## MD5SUM FUNCTION ##
function checkFileIntegrity() {
	case ${1} in
		app.jffs2)
			MD5SUM=${MD5SUM_APPFS}
			;;
		conf.jffs2)
			MD5SUM=${MD5SUM_CONFFS}
			;;
		root.jffs2)
			MD5SUM=${MD5SUM_ROOTFS}
			;;
		userapps.jffs2)
			MD5SUM=${MD5SUM_USERAPPFS}
			;;
		*)
			actOnIssue "Unknown file"
			;;
	esac
	if [ ! "$(/usr/bin/md5sum ${1} | /usr/bin/awk '{ print $1; }')" = "${MD5SUM}" ]; then
		actOnIssue "md5sum does not match (file=${1}|downloaded=$(/usr/bin/md5sum ${1} | /usr/bin/awk '{ print $1; }')|required=${MD5SUM})"
	fi
}

## CHECK FILES ON TFTP SERVER ##
cd /tmp
for file in "root.jffs2" "app.jffs2" "userapps.jffs2" "conf.jffs2"; do
	if ! /bin/echo "get ${REMOTEPATH}/${file}" | ${TFTP} ${REMOTEIP} &>/dev/null; then
		actOnIssue "Downloading '${file}' file failed"
	elif [ ! -f /tmp/${file} ]; then
		actOnIssue "'${file}' file missing"
	fi
	## check file integrity
	checkFileIntegrity ${file}
	/bin/rm -f /tmp/${file}
done

## MAIN ##

## SET MTD1 ##
/bin/mount -o remount,rw /
if [ -c /dev/mtd1 ]; then
	if ! /bin/rm -f /dev/mtd1 &>/dev/null; then
		actOnIssue "Removing /dev/mtd1 failed"
	fi
fi
if ! /bin/mknod /dev/mtd1 c 90 2 &>/dev/null; then
	actOnIssue "Making /dev/mtd1 failed"
fi
/bin/mount -o remount,ro /

## update prep_phy
PREPPHY='run prep_phy; '
if [ "$(/bin/cat /sys/class/beronet/gateway/li_count)" = "2" ]; then
	PREPPHY=''
else
	updateUboot "watchdog run prep_phy; mw.l 0x11400000 30000000" "restoring watchdog failed"
	updateUboot "prep_phy mw 0x100DE358 0x43; mw 0x100DE35C 0xCC1" "restoring prep_phy failed"
fi

## fetch ssh-password
PASSWORD=$(/bin/cat /etc/shadow | /bin/grep root | /usr/bin/cut -d ':' -f 2)
updateUboot "password ${PASSWORD}" "Storing current password failed"

## update u-boot
updateUboot "ipaddr ${LOCALIP}" "Updating the Local IP-Address failed"
updateUboot "serverip ${REMOTEIP}" "Updating the Server IP-Address failed"
updateUboot "inst3 erase 1:35-86; tftp 1000000 ${REMOTEPATH}/root.jffs2; cp.b 1000000 11460000 \${filesize}" "Updating remote path failed (root.jffs2)"
updateUboot "inst4 erase 1:87-162; tftp 1000000 ${REMOTEPATH}/app.jffs2; cp.b 1000000 11a60000 \${filesize}" "Updating remote path failed (app.jffs2)"
updateUboot "inst5 erase 1:163-226; tftp 1000000 ${REMOTEPATH}/userapps.jffs2; cp.b 1000000 12460000 \${filesize}" "Updating remote path failed (userapps.jffs2)"
updateUboot "inst6 erase 1:227-254; tftp 1000000 ${REMOTEPATH}/conf.jffs2; cp.b 1000000 12c60000 \${filesize}" "Updating remote path failed (conf.jffs2)"
updateUboot "boot_ins ${PREPPHY}run inst0; run inst3; run inst4; run inst5; run inst6; run inst7; run inst8; run inst9" "Updating boot_ins failed"

## last update
updateUboot "reproduced 1" "Setting reproduced flag failed"
updateUboot "bootcmd run boot_ins" "Updating bootcmd command failed"

## successful
if [ -f /tmp/reproduce.gui ]; then
	## from gui. reboot
	/sbin/reboot &>/dev/null &
else
	## from bfdetect. give info
	/bin/touch /tmp/reproduce.successful
fi

exit 0
